<?php

namespace App\Models\Location;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use App\Models\Job\Job;
use App\Models\User\User;
use App\Models\Classified\Classified;
use App\Models\Project\Project;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;

class Country extends Model
{
    use HasSlug, HasFactory, HasUuidTrait, CreatedByTrait;

    protected $fillable = [
        'uuid',
        'country_name',
        'country_code',
        'slug',
        'created_by'
    ];

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('country_name')->saveSlugsTo('slug');
    }

    public function states(): HasMany
    {
        return $this->hasMany(State::class);
    }

    public function cities(): HasMany
    {
        return $this->hasMany(City::class);
    }

    public function jobs(): HasMany
    {
        return $this->hasMany(Job::class);
    }

    public function classifieds(): HasMany
    {
        return $this->hasMany(Classified::class);
    }

    public function projects(): HasMany
    {
        return $this->hasMany(Project::class);
    }

    /**
     * Get the count of users in this country through user_locations
     * This is optimized to only return the count, not load the actual users
     */
    public function users()
    {
        return $this->hasManyThrough(
            User::class,
            \App\Models\User\UserLocation::class,
            'country_id',
            'id',
            'id',
            'user_id'
        )->selectRaw('COUNT(DISTINCT users.id) as user_count');
    }
    /**
     * Get users with the 'Service' role in this country
     */
    public function members(): HasManyThrough
    {
        return $this->hasManyThrough(
            User::class,
            \App\Models\User\UserLocation::class,
            'country_id',
            'id',
            'id',
            'user_id'
        )->whereHas('roles', function ($query) {
            $query->where('name', 'Service');
        });
    }
}
