<?php

declare(strict_types=1);

namespace App\Models\Membership;

use App\Models\User\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use App\Models\User\MyRole;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use App\Enums\MembershipPaymentTerm;
use App\Enums\MembershipType;
use App\Models\Membership\Traits\MembershipPlanHelperTrait;
use App\Observers\MembershipPlanObserver;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
#[ObservedBy(MembershipPlanObserver::class)]
class MembershipPlan extends Model implements Auditable
{
    use HasFactory, SoftDeletes, HasUuidTrait, CreatedByTrait, HasSlug, AuditableTrait, MembershipPlanHelperTrait;

    protected $fillable = [
        'uuid',
        'role_id',
        'package_title',
        'slug',
        'package_description',
        'membership_type',
        'price_monthly',
        'price_quarterly',
        'price_half_yearly',
        'price_yearly',
        'preferred_payment_term',
        'is_active',
        'is_visible',
        'is_featured',
        'is_best_value',
        'is_most_popular',
        'created_by',
        'trial_days',
        'has_trial',
        'display_order',
        'features',
        'max_users',
        'minimum_term_months',
    ];

    protected $casts = [
        'price_monthly' => 'decimal:2',
        'price_quarterly' => 'decimal:2',
        'price_half_yearly' => 'decimal:2',
        'price_yearly' => 'decimal:2',
        'is_active' => 'boolean',
        'is_visible' => 'boolean',
        'is_featured' => 'boolean',
        'has_trial' => 'boolean',
        'features' => 'array',
        'preferred_payment_term' => MembershipPaymentTerm::class,
        'membership_type' => MembershipType::class,
    ];

    protected $auditInclude = [
        'package_title',
        'package_description',
        'membership_type',
        'price_monthly',
        'price_quarterly',
        'price_half_yearly',
        'price_yearly',
        'preferred_payment_term',
        'is_active',
        'is_visible',
        'is_featured',
        'created_by',
        'trial_days',
        'has_trial',
        'display_order',
        'features',
        'max_users',
        'minimum_term_months',
    ];
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('package_title')->saveSlugsTo('slug')->doNotGenerateSlugsOnUpdate();
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function additionalCosts(): HasMany
    {
        return $this->hasMany(MembershipPlanAdditionalCost::class);
    }

    public function packageUserInfo(): HasOne
    {
        return $this->hasOne(PackageUserInfo::class);
    }

    public function planModules(): HasMany
    {
        return $this->hasMany(PlanModule::class);
    }

    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    public function role(): BelongsTo
    {
        return $this->belongsTo(MyRole::class);
    }
}
