<?php

declare(strict_types=1);

namespace App\Models\Membership;

use App\Models\User\User;
use App\Enums\PaymentType;
use App\Enums\PaymentMethod;
use App\Enums\PaymentStatus;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use Illuminate\Database\Eloquent\SoftDeletes;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;

class Payment extends Model implements Auditable
{
    use HasFactory, HasUuidTrait, CreatedByTrait, SoftDeletes, AuditableTrait;

    protected $fillable = [
        'uuid',
        'purchase_id',
        'user_id',
        'payment_type',
        'payment_method',
        'amount',
        'currency',
        'status',
        'transaction_id',
        'service_response',
        'payment_date',
        'notes',
        'metadata',
        'created_by',
    ];

    protected $casts = [
        'payment_type' => PaymentType::class,
        'payment_method' => PaymentMethod::class,
        'status' => PaymentStatus::class,
        'amount' => 'decimal:2',
        'payment_date' => 'datetime',
        'service_response' => 'array',
        'metadata' => 'array',
    ];

    protected $auditInclude = [
        'purchase_id',
        'user_id',
        'payment_method',
        'amount',
        'status',
        'payment_date',
    ];

    public function purchase(): BelongsTo
    {
        return $this->belongsTo(Purchase::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function isSuccessful(): bool
    {
        return $this->status === PaymentStatus::Paid;
    }

    public function isPending(): bool
    {
        return $this->status === PaymentStatus::Pending;
    }

    public function isFailed(): bool
    {
        return $this->status === PaymentStatus::Failed;
    }

    public function isRefunded(): bool
    {
        return $this->status === PaymentStatus::Refunded;
    }
}
