<?php

declare(strict_types=1);

namespace App\Models\Membership;

use App\Models\User\User;
use App\Enums\PurchaseItemType;
use App\Models\Membership\MembershipPlan;
use App\Models\Membership\Subscription;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use Illuminate\Database\Eloquent\SoftDeletes;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;

class Purchase extends Model implements Auditable
{
    use HasFactory, HasUuidTrait, CreatedByTrait, SoftDeletes, AuditableTrait;

    protected $fillable = [
        'uuid',
        'user_id',
        'item_type',
        'purchaseable_type',
        'purchaseable_id',
        'expire_date',
        'amount',
        'currency',
        'quantity',
        'total_amount',
        'tax_amount',
        'discount_amount',
        'status',
        'purchase_date',
        'notes',
        'metadata',
        'created_by',
    ];

    protected $casts = [
        'item_type' => PurchaseItemType::class,
        'amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'quantity' => 'integer',
        'purchase_date' => 'datetime',
        'metadata' => 'array',
        'status' => 'boolean',
    ];

    protected $auditInclude = [
        'user_id',
        'item_type',
        'amount',
        'total_amount',
        'status',
        'purchase_date',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function purchaseable(): MorphTo
    {
        return $this->morphTo();
    }

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    public function latestPayment(): ?Payment
    {
        return $this->payments()->latest()->first();
    }

    public function successfulPayments(): HasMany
    {
        return $this->hasMany(Payment::class)->where('status', 'success');
    }

    public function getTotalPaidAmount(): float
    {
        return $this->successfulPayments()->sum('amount');
    }

    public function isFullyPaid(): bool
    {
        return $this->getTotalPaidAmount() >= $this->total_amount;
    }

    public function getRemainingAmount(): float
    {
        return max(0, $this->total_amount - $this->getTotalPaidAmount());
    }

    /**
     * Get the membership plan if purchaseable is a subscription
     */
    public function getMembershipPlan(): ?MembershipPlan
    {
        if ($this->purchaseable_type === Subscription::class) {
            return $this->purchaseable?->membershipPlan;
        }
        return null;
    }

    /**
     * Get purchaseable details for display
     */
    public function getPurchaseableDetails(): string
    {
        if ($this->purchaseable_type === Subscription::class) {
            return $this->purchaseable?->membershipPlan?->package_title ?? 'Unknown Plan';
        }
        return $this->purchaseable_type ? class_basename($this->purchaseable_type) : 'N/A';
    }

    // Scopes
    public function scopeWithMembershipPlan($query, string $planSlug)
    {
        return $query->whereHas('purchaseable.membershipPlan', function ($subQuery) use ($planSlug) {
            $subQuery->where('slug', $planSlug);
        });
    }

    public function scopeWithActiveMembershipPlans($query)
    {
        return $query->whereHas('purchaseable.membershipPlan', function ($subQuery) {
            $subQuery->where('is_active', true);
        });
    }

    public function scopeWithFeaturedMembershipPlans($query)
    {
        return $query->whereHas('purchaseable.membershipPlan', function ($subQuery) {
            $subQuery->where('is_featured', true);
        });
    }
}
