<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\Traits\CreatedByTrait;
use Illuminate\Database\Eloquent\SoftDeletes;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use RalphJSmit\Laravel\SEO\Support\SEOData;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use App\Enums\PageStatus;
use Biostate\FilamentMenuBuilder\Traits\Menuable;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Page extends Model implements HasMedia
{
    use HasSlug, CreatedByTrait, SoftDeletes, HasSEO, InteractsWithMedia, Menuable;

    protected $fillable = [
        'title',
        'heading',
        'slug',
        'content',
        'data',
        'parent_id',
        'page_type',
        'route_name',
        'template',
        'published_at',
        'expired_at',
        'status',
        'template',
    ];

    protected $casts = [
        'data' => 'array',
        'status' => PageStatus::class,
        'content' => 'array',
    ];

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('title')
            ->saveSlugsTo('slug')
            ->doNotGenerateSlugsOnUpdate();
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('featured_image')
            ->singleFile()
            ->useFallbackUrl('/images/image_placeholder.webp')
            ->useFallbackPath(public_path('/images/image_placeholder.webp'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->width(368)
                    ->height(232)
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->width(768)
                    ->height(480)
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->width(1024)
                    ->height(768)
                    ->sharpen(10);
            });
    }

    public function getDynamicSEOData(): SEOData
    {
        $pathToFeaturedImageRelativeToPublicPath = $this->getFirstMediaUrl('featured_image');
        $seoTitle = (!empty($this->seo->title)) ? $this->seo->title : $this->title;
        $seoDescription = (!empty($this->seo->description)) ? $this->seo->description : '';
        // Override only the properties you want:
        return new SEOData(
            title: $seoTitle,
            description: $seoDescription,
            image: $pathToFeaturedImageRelativeToPublicPath,
        );
    }

    public function getMenuLinkAttribute(): string
    {
        return url($this->slug);
    }

    public function getMenuNameAttribute(): string
    {
        return $this->name;
    }

    public function getContentTextAttribute(): string
    {
        return tiptap_converter()->asText($this->content);
    }

    public function getContentHtmlAttribute(): string
    {
        return tiptap_converter()->asHTML($this->content);
    }

    public function widgets(): BelongsToMany
    {
        return $this->belongsToMany(Widget::class, 'page_widget')
            ->withPivot(['sort_order', 'enable'])
            ->withTimestamps();
    }

    public function pageWidgets(): HasMany
    {
        return $this->hasMany(PageWidget::class);
    }
}
