<?php

declare(strict_types=1);

namespace App\Models\Product;

use App\Models\User\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Carbon;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use App\Contracts\Previewable;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use App\Filament\Traits\HasSeoSection;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use App\Filament\Traits\Menuable;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\Product\ProductCategory;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use RalphJSmit\Laravel\SEO\Support\SEOData;
use RalphJSmit\Laravel\SEO\SchemaCollection;
use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class Product extends Model implements HasMedia, Auditable
{
    use HasFactory, SoftDeletes, InteractsWithMedia, HasSEO, HasSlug, HasUuidTrait, CreatedByTrait, HasSEO, AuditableTrait;

    protected $fillable = [
        'uuid',
        'parent_id',
        'product_title',
        'slug',
        'product_description',
        'product_sku',
        'link',
        'quantity',
        'price',
        'created_by',
        'published_at',
        'expired_at',
    ];

    protected $casts = [
        'quantity' => 'integer',
        'price' => 'decimal:2',
        'published_at' => 'datetime',
        'expired_at' => 'datetime',
    ];

    protected $auditInclude = [
        'product_title',
        'product_description',
        'product_sku',
        'link',
        'slug',
        'quantity',
        'price',
        'created_by',
        'published_at',
        'expired_at',
    ];

    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(
            ProductCategory::class,
            'product_product_category',
            'product_id',
            'product_category_id'
        );
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function scopePublished($query)
    {
        return $query->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->where(function ($query) {
                $query->whereNull('expired_at')
                    ->orWhere('expired_at', '>', now());
            });
    }

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('product_title')->saveSlugsTo('slug')->doNotGenerateSlugsOnUpdate();
    }

    public function getDynamicSEOData(): SEOData
    {
        $pathToFeaturedImageRelativeToPublicPath = $this->getFirstMediaUrl('featured_image');

        $seoTitle = empty($this->seo->title) ? $this->product_title : $this->seo->title;
        $seoDescription = empty($this->seo->description) ? $this->product_description : $this->seo->description;

        // Override only the properties you want:
        return new SEOData(
            title: $seoTitle,
            description: $seoDescription,
            image: $pathToFeaturedImageRelativeToPublicPath,
            author: $this->author->name,
            schema: SchemaCollection::make()->addArticle(),
        );
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('gellery_images')
            ->useFallbackUrl('/images/image_placeholder.webp')
            ->useFallbackPath(public_path('/images/image_placeholder.webp'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Crop, 285, 300)
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->width(768)
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->width(1024)
                    ->height(768)
                    ->sharpen(10);
            });
    }
} 