<?php

declare(strict_types=1);

namespace App\Models\Product;

use App\Models\User\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Carbon;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use App\Contracts\Previewable;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use App\Filament\Traits\HasSeoSection;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use RalphJSmit\Laravel\SEO\Support\SEOData;
use App\Filament\Traits\Menuable;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\Product\Product;
use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use RalphJSmit\Laravel\SEO\SchemaCollection;

class ProductCategory extends Model implements HasMedia, Auditable
{
    use HasFactory, SoftDeletes, InteractsWithMedia, HasSEO, HasSlug, HasUuidTrait, CreatedByTrait, HasSEO, AuditableTrait;

    protected $fillable = [
        'uuid',
        'parent_id',
        'product_category_name',
        'slug',
        'product_category_description',
        'is_active',
        'created_by',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    public function parent(): BelongsTo
    {
        return $this->belongsTo(self::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(self::class, 'parent_id');
    }

    public function products(): BelongsToMany
    {
        return $this->belongsToMany(
            Product::class,
            'product_product_category',
            'product_category_id',
            'product_id'
        );
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }
    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('product_title')->saveSlugsTo('slug')->doNotGenerateSlugsOnUpdate();
    }

    public function getDynamicSEOData(): SEOData
    {
        $pathToFeaturedImageRelativeToPublicPath = $this->getFirstMediaUrl('featured_image');

        $seoTitle = empty($this->seo->title) ? $this->product_category_name : $this->seo->title;
        $seoDescription = empty($this->seo->description) ? $this->product_category_description : $this->seo->description;

        // Override only the properties you want:
        return new SEOData(
            title: $seoTitle,
            description: $seoDescription,
            image: $pathToFeaturedImageRelativeToPublicPath,
            author: $this->author->name,
            schema: SchemaCollection::make()->addArticle(),
        );
    }
} 