<?php

namespace App\Models\Project;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use App\Models\Team\TeamMember;
use App\Models\User\User;
use App\Enums\ProjectStatus;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Enums\ProjectCompletedStatus;
use Spatie\Image\Enums\Fit;
use Illuminate\Database\Eloquent\Casts\AsStringable;
use App\Models\Project\Trait\ProjectHelperTrait;
use App\Models\Location\City;
use App\Models\Location\State;
use App\Models\Location\Country;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

class Project extends Model implements HasMedia, Auditable
{
    use HasSlug, HasUuidTrait, CreatedByTrait, SoftDeletes, InteractsWithMedia, HasFactory, HasSEO, AuditableTrait, ProjectHelperTrait;

    protected $fillable = [
        'user_id',
        'title',
        'parent_id',
        'slug',
        'description',
        'address',
        'location',
        'city_id',
        'state_id',
        'country_id',
        'postal_code',
        'latitude',
        'longitude',
        'techniques',
        'equipments',
        'introduction',
        'objective',
        'challenges',
        'solutions',
        'outcomes',
        'completed_status',
        'project_status',
        'completion_date',
        'project_cost',
        'tags',
        'metrics',
        'created_by',
        'published_at',
    ];

    protected $casts = [
        'project_status' => ProjectStatus::class,
        'published_at' => 'datetime',
        'address' => AsStringable::class,
        'completed_status' => ProjectCompletedStatus::class,
        'metrics' => 'array',
        'completion_date' => 'date',
        'project_cost' => 'integer',
        'tags' => 'array',
    ];

    protected $auditInclude = [
        'title',
        'slug',
        'description',
        'address',
        'techniques',
        'equipments',
    ];

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('title')->saveSlugsTo('slug')->doNotGenerateSlugsOnUpdate();
    }

    public function types(): BelongsToMany
    {
        return $this->belongsToMany(ProjectType::class, 'project_project_type');
    }

    public function team(): BelongsToMany
    {
        return $this->belongsToMany(TeamMember::class, 'project_team')
            ->withTimestamps()
            ->withPivot(['role', 'order']);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function city(): BelongsTo
    {
        return $this->belongsTo(City::class);
    }

    public function state(): BelongsTo
    {
        return $this->belongsTo(State::class);
    }

    public function country(): BelongsTo
    {
        return $this->belongsTo(Country::class);
    }

    public function favoritesUsers(): MorphToMany
    {
        return $this->morphToMany(User::class, 'model');
    }

    public function scopePublished($query)
    {
        return $query->where('project_status', ProjectStatus::Published->value)
            ->whereNotNull('published_at');
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('featured_image')
            ->singleFile()
            ->useFallbackUrl('/images/image_placeholder.webp')
            ->useFallbackPath(public_path('/images/image_placeholder.webp'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 368)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 768)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->fit(Fit::Contain, 1024)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });

        $this->addMediaCollection('gallery')
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 318)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 600)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->fit(Fit::Contain, 1920)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });
    }


    public function getMapAddressAttribute(): array
    {
        return [
            "lat" => (float)$this->latitude,
            "lng" => (float)$this->longitude,
        ];
    }

    /**
     * Takes a Google style Point array of 'lat' and 'lng' values and assigns them to the
     * 'latitude' and 'longitude' attributes on this model.
     *
     * Used by the Filament Google Maps package.
     *
     * Requires the 'map_address' attribute be included in this model's $fillable array.
     *
     * @param ?array $location
     * @return void
     */
    public function setMapAddressAttribute(?array $location): void
    {
        if (is_array($location)) {
            $this->attributes['latitude'] = $location['lat'];
            $this->attributes['longitude'] = $location['lng'];
            unset($this->attributes['map_address']);
        }
    }

    /**
     * Get the lat and lng attribute/field names used on this table
     *
     * Used by the Filament Google Maps package.
     *
     * @return string[]
     */
    public static function getLatLngAttributes(): array
    {
        return [
            'lat' => 'latitude',
            'lng' => 'longitude',
        ];
    }

    /**
     * Get the name of the computed location attribute
     *
     * Used by the Filament Google Maps package.
     *
     * @return string
     */
    public static function getComputedLocation(): string
    {
        return 'map_address';
    }
}
