<?php

namespace App\Models\Project\Trait;

use Illuminate\Support\Facades\Auth;

trait ProjectHelperTrait
{
    public function getPermalinkAttribute(): string
    {
        return route('project.show', $this->slug);
    }

    public function getProjectStatus()
    {
        if (!empty($this->project_status)) {
            return $this->project_status->getLabel();
        }
        return ''; // Default status if not set
    }

    /**
     * Get the user's address based on the specified type.
     *
     * @param string $type The type of address to retrieve (default is 'full').
     * @return ?string The formatted address string, or null if no address is found.
     */
    public function address($type = 'full'): ?string
    {
        // Return location column for full type
        if ($type === 'full') {
            return $this->location ?? '';
        }

        $addressParts = [];

        if ($this->city()->exists()) {
            $addressParts['city'] = $this->city->city_name;
        }

        if ($this->state()->exists()) {
            $addressParts['state'] = $this->state->state_code;
        }

        if ($this->country()->exists()) {
            $addressParts['country'] = $this->country->country_name;
        }

        // Return empty if no parts exist
        if (empty($addressParts)) {
            return '';
        }

        // For short type, return "City, State Code, Country"
        if ($type === 'short') {
            $formattedAddress = [];
            if (isset($addressParts['city'])) {
                $formattedAddress[] = $addressParts['city'];
            }
            if (isset($addressParts['state'])) {
                $formattedAddress[] = $addressParts['state'];
            }
            if (isset($addressParts['country'])) {
                $formattedAddress[] = $addressParts['country'];
            }
            return implode(', ', $formattedAddress);
        }

        // For area type, return "City Postal Code, State Code"
        if ($type === 'area') {
            $formattedAddress = [];
            if (isset($addressParts['city'])) {
                $formattedAddress[] = $addressParts['city'];
            }
            if ($this->postal_code) {
                $formattedAddress[] = $this->postal_code;
            }
            if (isset($addressParts['state'])) {
                $formattedAddress[] = $addressParts['state'];
            }
            return implode(' ', $formattedAddress);
        }

        return '';
    }

    public function getMapUrlAttribute(): string
    {
        $location = $this->latitude . ',' . $this->longitude;

        return "https://www.google.com/maps?q={$location}";
    }

    public function mapIframeUrl($settings = []): ?string
    {
        $zoomLevel = $settings['zoom_level'] ?? 15;
        $mapType   = $settings['map_type'] ?? 'roadmap';
        $location  = $this->address;
        if (! $location) {
            return null;
        }

        if (! $this->latitude || ! $this->longitude) {
            return null;
        }

        $coords = $this->latitude . ',' . $this->longitude;
        $mapUrl = 'https://www.google.com/maps/embed/v1/place?key=' . config('services.google.maps_api_key', '') . "&q={$coords}" . '&zoom=' . $zoomLevel . '&maptype=' . $mapType;
        return $mapUrl;
    }

    public function getProjectTags($limit = 9)
    {
        $tags = !empty($this->tags) ? array_slice($this->tags, 0, $limit) : [];
        return $tags;
    }

    public function getProjectCategory($limit = 1)
    {
        $categories = [];
        if ($this->types()->exists()) {
            $categories = $this->types()->take($limit);
        }
        return $categories;
    }

    public function publishedAt()
    {
        $published_at = '';
        if ($this->published_at) {
            $published_at = $this->published_at->format('d F, Y');
        } else {
            $published_at = $this->created_at->format('d F, Y');
        }
        return $published_at;
    }

    public function getProjectCost()
    {
        $projectStatus = $this->completed_status;
        $cost = null;
        if ($projectStatus) {
            if ($projectStatus->value === 'completed' && !empty($this->project_cost)) {
                $cost = $this->project_cost;
            }
        }
        return $cost;
    }


    public function isFavorite(): bool
    {
        if (!Auth::check()) {
            return false;
        }
        return Auth::user()->favoriteProject()->where('model_id', $this->id)->exists();
    }
}
