<?php

namespace App\Models;

use App\Models\Traits\HasUuidTrait;
use App\Models\User\User;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Image\Enums\Fit;

class Review extends Model implements HasMedia
{
    use HasFactory, HasUuidTrait, SoftDeletes, InteractsWithMedia;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'uuid',
        'review_rating',
        'review_title',
        'review_content',
        'review_user_id',
        'reviewable_type',
        'reviewable_id',
        'status',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'review_rating' => 'integer',
        'status'        => 'boolean',
    ];

    /**
     * Get the user who wrote the review.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'review_user_id');
    }

    /**
     * Get the reviewable model that the review belongs to.
     */
    public function reviewable(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * Get the ratings for the review.
     */
    public function ratings(): HasMany
    {
        return $this->hasMany(ReviewRating::class);
    }

    /**
     * Get the average of all rating types for this review.
     */
    public function getAverageRatingAttribute(): float
    {
        if ($this->ratings->isEmpty()) {
            return 0;
        }

        return round($this->ratings->avg('rating_value'), 1);
    }

    public function getReviewCreatedTimeFormatAttribute(): string
    {
        $time = '';
        if (! empty($this->created_at)) {
            $time = Carbon::parse($this->created_at)->diffForHumans();
            if ($time == '0 seconds ago') {
                $time = 'Just Now';
            }
        }
        return $time;
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('review_attachments')
            ->useFallbackUrl('/images/review-attachment.jpeg')
            ->useFallbackPath(public_path('images/review-attachment.jpeg'))
            ->acceptsMimeTypes(['image/jpeg', 'image/png', 'image/jpg', 'image/webp', 'application/pdf'])
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Max, desiredWidth: 150, desiredHeight: 150);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Max, desiredWidth: 500, desiredHeight: 500);
            });
    }
}
