<?php

declare(strict_types=1);

namespace App\Models\Team;

use App\Models\User\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use App\Enums\TeamMemberStatus;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Image\Enums\CropPosition;
use Spatie\Image\Enums\Fit;

class TeamMember extends Model implements HasMedia
{
    use HasFactory;
    use SoftDeletes;
    use HasUuidTrait;
    use CreatedByTrait;
    use InteractsWithMedia;

    protected $fillable = [
        'uuid',
        'user_id',
        'first_name',
        'last_name',
        'team_member_role',
        'phone',
        'email',
        'experience_years',
        'credentials',
        'description',
        'address',
        'social_links',
        'team_member_status',
        'published_at',
        'created_by'
    ];

    protected $casts = [
        'credentials' => 'array',
        'social_links' => 'array',
        'team_member_status' => TeamMemberStatus::class,
    ];

    protected $auditInclude = [
        'team_member_status',
        'published_at',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getFullNameAttribute(): string
    {
        if (empty($this->first_name) && empty($this->last_name)) {
            return 'Team Member';
        }
        if (empty($this->last_name)) {
            return $this->first_name;
        }
        return $this->first_name . ' ' . $this->last_name;
    }

    // Define media collections
    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('featured_image')
            ->singleFile()
            ->useFallbackUrl('/images/image_placeholder.webp')
            ->useFallbackPath(public_path('/images/image_placeholder.webp'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Crop, 285, 300)
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->width(768)
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->width(1024)
                    ->height(768)
                    ->sharpen(10);
            });
    }
}
