<?php

namespace App\Models\User;

use App\Enums\DayOfWeek;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;

class BusinessHour extends Model
{
    protected $fillable = [
        'user_id',
        'day_of_week',
        'is_open',
        'hours_label',
        'open_time',
        'close_time'
    ];

    protected $casts = [
        'day_of_week' => DayOfWeek::class,
        'is_open' => 'boolean',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function scopeOpen(Builder $query): Builder
    {
        return $query->where('is_open', true);
    }

    public function scopeForDay(Builder $query, string $day): Builder
    {
        return $query->where('day_of_week', $day);
    }

    public function scopeCurrentlyOpen(Builder $query): Builder
    {
        $now = Carbon::now();
        $currentDay = strtolower($now->format('l'));
        $currentTime = $now->format('H:i');

        return $query->where('day_of_week', $currentDay)
                    ->where('is_open', true)
                    ->where('open_time', '<=', $currentTime)
                    ->where('close_time', '>=', $currentTime);
    }

    public function isCurrentlyOpen(): bool
    {
        if (!$this->is_open) {
            return false;
        }

        $now = Carbon::now();
        $currentDay = strtolower($now->format('l'));
        $currentTime = $now->format('H:i');

        return $this->day_of_week->value === $currentDay
            && $this->open_time <= $currentTime
            && $this->close_time >= $currentTime;
    }

    public function getFormattedHoursAttribute(): string
    {
        if (!$this->is_open) {
            return 'Closed';
        }

        return date('g:i A', strtotime($this->open_time)) . ' - ' . date('g:i A', strtotime($this->close_time));
    }
}