<?php

declare(strict_types=1);

namespace App\Models\User;

use App\Enums\InquiryStatus;
use App\Enums\InquiryType;
use App\Models\Traits\HasUuidTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class Inquiry extends Model
{
    use HasFactory, HasUuidTrait;

    protected $fillable = [
        'uuid',
        'contact_id',
        'user_id',
        'form_data',
        'inquiry_status',
        'inquiry_type',
        'inquiryable_type',
        'inquiryable_id',
    ];

    protected $casts = [
        'form_data' => 'array',
        'inquiry_status' => InquiryStatus::class,
        'inquiry_type' => InquiryType::class,
    ];

    // Only timestamps, no updated_at according to migration
    const UPDATED_AT = null;

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function inquiryable(): MorphTo
    {
        return $this->morphTo();
    }

    // Scope for filtering by inquiry status
    public function scopeByStatus($query, InquiryStatus $status)
    {
        return $query->where('inquiry_status', $status);
    }

    // Scope for filtering by inquiry type
    public function scopeByType($query, InquiryType $type)
    {
        return $query->where('inquiry_type', $type);
    }

    // Check if inquiry is pending
    public function isPending(): bool
    {
        return $this->inquiry_status === InquiryStatus::Pending;
    }

    // Check if inquiry is contacted
    public function isContacted(): bool
    {
        return $this->inquiry_status === InquiryStatus::Contacted;
    }
}
