<?php

namespace App\Models\User\Traits;

use Illuminate\Support\Number;
use App\Enums\MembershipType;
use App\Enums\MemberRatingType;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Auth;

trait UserHelperTrait
{

    public function getPermalinkAttribute(): string
    {
        return route('member.show', $this->slug);
    }

    public function memberName(): string
    {
        return $this->profile()->exists() ? $this->profile->company_name ?? $this->name : $this->name;
    }

    public function parentCategories($limit = 3)
    {
        return $this->memberCategories()->whereNull('parent_id')->take($limit)->get();
    }

    /**
     * Get the user's address based on the specified type.
     *
     * @param string $type The type of address to retrieve (default is 'full').
     * @return ?string The formatted address string, or null if no address is found.
     */
    public function address($type = 'full'): ?string
    {
        $location = $this->primaryLocation;
        if (! $location) {
            return null;
        }

        $addressParts = [];
        $stateCode = null;

        if ($location->address_line_1) {
            $addressParts['address_line_1'] = $location->address_line_1;
        }

        if ($location->city()->exists()) {
            $addressParts['city'] = $location->city->city_name;
        }

        if ($location->state()->exists()) {
            $addressParts['state'] = $location->state->state_name;
            $stateCode = $location->state->state_code;
        }

        if ($location->country()->exists()) {
            $addressParts['country'] = $location->country->country_name;
        }
        if ($location->postal_code) {
            $addressParts['postal_code'] = $location->postal_code;
        }

        if ($type == 'short') {
            unset($addressParts['address_line_1']);
            unset($addressParts['postal_code']);
        }
        if ($type == 'area') {
            unset($addressParts['address_line_1']);
            unset($addressParts['country']);
            unset($addressParts['postal_code']);

            if ($stateCode) {
                unset($addressParts['state']);
                $addressParts['state'] = $stateCode;
            }
        }


        return ! empty($addressParts) ? implode(', ', $addressParts) : '';
    }

    public function getMapUrlAttribute(): string
    {
        $location = $this->primaryLocation;
        if (! $location) {
            return '';
        }

        return "https://www.google.com/maps?q={$location->latitude},{$location->longitude}";
    }

    public function mapIframeUrl($settings = []): ?string
    {
        $zoomLevel = $settings['zoom_level'] ?? 15;
        $mapType   = $settings['map_type'] ?? 'roadmap';
        $location  = $this->primaryLocation;
        if (! $location) {
            return null;
        }

        if (! $location->latitude || ! $location->longitude) {
            return null;
        }

        $coords = $location->latitude . ',' . $location->longitude;
        $mapUrl = 'https://www.google.com/maps/embed/v1/place?key=' . config('services.google.maps_api_key', '') . "&q={$coords}" . '&zoom=' . $zoomLevel . '&maptype=' . $mapType;

        return $mapUrl;
    }

    public function logoImage(): ?string
    {
        if ($this->getMedia('logo')->count() > 0) {
            $imageUrl = $this->getFirstMediaUrl('logo', 'small');
        } else {
            $imageUrl = $this->getFirstMediaUrl('avatar', 'small');
        }

        return $imageUrl;
    }

    public function ratingCount($startDate = null, $endDate = null): array
    {
        $query = $this->reviews();

        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }

        $avgRating = $query->avg('review_rating') ?? 0;
        $avgRating = Number::format($avgRating, 1);
        $ratingCount = $query->count();

        return [
            'rating' => $avgRating,
            'count' => $ratingCount,
        ];
    }

    public function getMembershipTypeAttribute(): MembershipType
    {
        return $this->currentMembershipPlan()?->membership_type ?? MembershipType::Free;
    }

    public function memberAbout(): string
    {
        return $this->profile()->exists() ? html_entity_decode(strip_tags($this->profile?->about), ENT_QUOTES | ENT_HTML5, 'UTF-8') : '';
    }


    /**
     * Generate comprehensive review statistics for a user
     *
     * @param int $userId The ID of the user to get statistics for
     * @return array Review statistics including total count, average ratings, and distribution
     */
    public function getUserReviewsStats(): array
    {
        $user = $this;

        if (!$user) {
            return [
                'total_reviews' => 0,
                'average_rating' => 0,
                'rating_distribution' => [],
                'rating_types' => [],
            ];
        }

        // Load reviews with their ratings
        $reviews = $user->reviews()->with('ratings')->where('status', true)->get();

        // If no reviews, return empty stats
        if ($reviews->isEmpty()) {
            return [
                'total_reviews' => 0,
                'average_rating' => 0,
                'rating_distribution' => [
                    5 => ['count' => 0, 'percentage' => 0],
                    4 => ['count' => 0, 'percentage' => 0],
                    3 => ['count' => 0, 'percentage' => 0],
                    2 => ['count' => 0, 'percentage' => 0],
                    1 => ['count' => 0, 'percentage' => 0],
                ],
                'rating_types' => $this->getEmptyRatingTypes(),
            ];
        }

        // Calculate total reviews count
        $totalReviews = $reviews->count();

        // Calculate average overall rating
        $averageRating = round($reviews->avg('review_rating'), 1);

        // Calculate rating distribution (5★, 4★, 3★, etc.)
        $ratingDistribution = [
            5 => ['count' => 0, 'percentage' => 0],
            4 => ['count' => 0, 'percentage' => 0],
            3 => ['count' => 0, 'percentage' => 0],
            2 => ['count' => 0, 'percentage' => 0],
            1 => ['count' => 0, 'percentage' => 0],
        ];

        foreach ($reviews as $review) {
            $rating = $review->review_rating;
            if (isset($ratingDistribution[$rating])) {
                $ratingDistribution[$rating]['count']++;
            }
        }

        // Calculate percentages
        foreach ($ratingDistribution as $rating => &$data) {
            $data['percentage'] = $totalReviews > 0 ? round(($data['count'] / $totalReviews) * 100, 1) : 0;
        }

        // Calculate rating type statistics
        $ratingTypeStats = $this->calculateRatingTypeStats($reviews);

        return [
            'total_reviews' => $totalReviews,
            'average_rating' => $averageRating,
            'rating_distribution' => $ratingDistribution,
            'rating_types' => $ratingTypeStats,
        ];
    }


    /**
     * Calculate statistics for each rating type
     *
     * @param \Illuminate\Support\Collection $reviews Collection of reviews
     * @return array Statistics for each rating type
     */
    // private function calculateRatingTypeStats($reviews): array
    // {
    //     $ratingTypes = $this->getEmptyRatingTypes();

    //     // Process all ratings across all reviews
    //     foreach ($reviews as $review) {
    //         foreach ($review->ratings as $rating) {
    //             $type = $rating->rating_type->value;

    //             // Increment count and sum for the specific rating type
    //             $ratingTypes[$type]['count']++;
    //             $ratingTypes[$type]['sum'] += $rating->rating;
    //         }
    //     }

    //     // Calculate averages and percentages
    //     foreach ($ratingTypes as $type => &$stats) {
    //         if ($stats['count'] > 0) {
    //             $stats['average'] = round($stats['sum'] / $stats['count'], 1);
    //             $stats['percentage'] = round(($stats['average'] / 5) * 100);
    //         }
    //     }

    //     return $ratingTypes;
    // }

    /**
     * Get empty rating type statistics structure
     *
     * @return array Empty rating type statistics
     */
    private function getEmptyRatingTypes(): array
    {
        $ratingTypes = [];

        foreach (MemberRatingType::cases() as $type) {
            $ratingTypes[$type->value] = [
                'label' => $type->getLabel(),
                'count' => 0,
                'sum' => 0,
                'average' => 0,
                'percentage' => 0,
            ];
        }

        return $ratingTypes;
    }
    /**
     * Calculate statistics for each rating type
     *
     * @param \Illuminate\Support\Collection $reviews Collection of reviews
     * @return array Statistics for each rating type
     */
    private function calculateRatingTypeStats($reviews): array
    {
        $ratingTypes = $this->getEmptyRatingTypes();

        // Process all ratings across all reviews
        foreach ($reviews as $review) {
            foreach ($review->ratings as $rating) {
                $type = $rating->rating_type->value;

                // Increment count and sum for the specific rating type
                $ratingTypes[$type]['count']++;
                $ratingTypes[$type]['sum'] += $rating->rating;
            }
        }

        // Calculate averages and percentages
        foreach ($ratingTypes as $type => &$stats) {
            if ($stats['count'] > 0) {
                $stats['average'] = round($stats['sum'] / $stats['count'], 1);
                $stats['percentage'] = round(($stats['average'] / 5) * 100);
            }
        }

        return $ratingTypes;
    }

    public function getWhyChooseUs(): array
    {
        $whyChooseUs = $this->profile?->getMeta('why_choose_us') ?? [];
        $formattedPoints = [];

        foreach ($whyChooseUs as $point) {
            if (isset($point['key']) && isset($point['value'])) {
                $formattedPoints[] = [
                    'title' => $point['key'],
                    'value' => $point['value']
                ];
            }
        }

        return $formattedPoints;
    }

    public function getOurSupportServices(): array
    {
        return $this->profile?->getMeta('support_services') ?? [];
    }

    public function isFavorite(): bool
    {
        if (!Auth::check()) {
            return false;
        }
        return Auth::user()->favoriteMembers()->where('model_id', $this->id)->exists();
    }

    public function isVerified(): bool
    {
        $isVerified = true;
        return $isVerified;
    }

    public function getAreas(): Collection
    {
        return $this->primaryLocation?->areas()->get();
    }

    public function getFaqs(): array
    {
        return !empty($this->profile?->getMeta('faqs')) && collect($this->profile?->getMeta('faqs'))->filter(
            fn($faq) => !empty($faq['question']) && !empty($faq['answer'])
        )->isNotEmpty() ? $this->profile?->getMeta('faqs') : [];
    }
}
