<?php

namespace App\Models\User;

use App\Models\Blog\Post;
use App\Models\Classified\Classified;
use App\Models\Job\Job;
use App\Models\Lead\Lead;
use App\Models\Membership\Subscription;
use App\Models\Project\Project;
use App\Models\Team\TeamMember;
use App\Models\Equipment\Equipment;
use App\Models\Job\JobApplication;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Auth\MustVerifyEmail as MustVerifyEmailTrait;
use App\Notifications\ResetPassword as ResetPasswordNotification;
use App\Notifications\QueuedEmailVerification;
use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Permission\Traits\HasRoles;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use Kodeine\Metable\Metable;
use App\Models\User\Traits\UserHelperTrait;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use App\Models\Review;
use App\Models\Traits\HasUuidTrait;
use App\Enums\SubscriptionStatus;
use App\Models\Membership\MembershipPlan;
use Filament\Models\Contracts\FilamentUser;
use Filament\Panel;
use Laravel\Cashier\Billable;
use App\Models\User\UserVideo;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable implements HasMedia, FilamentUser, MustVerifyEmail
{
    /**
     * Send the password reset notification.
     *
     * @param  string  $token
     * @return void
     */
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new ResetPasswordNotification($token));
    }

    /**
     * Send the email verification notification.
     *
     * @return void
     */
    public function sendEmailVerificationNotification()
    {
        $this->notify(new QueuedEmailVerification());
    }
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, HasRoles, HasSlug, InteractsWithMedia, Notifiable, SoftDeletes, Metable, UserHelperTrait, HasUuidTrait, Billable, MustVerifyEmailTrait, HasApiTokens;

    protected $metaTable = 'users_meta'; //optional.
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'uuid',
        'name',
        'slug',
        'email',
        'password',
    ];

    // protected $meta_fields = [
    //     'faqs',
    // ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    public function canAccessPanel(Panel $panel): bool
    {
        return $this->hasAnyRole(['Admin', 'Super Admin']);
    }

    // Relationship
    public function primaryLocation(): HasOne
    {
        return $this->hasOne(UserLocation::class)->where('primary_location', 1);
    }

    public function profile(): HasOne
    {
        return $this->hasOne(UserProfile::class);
    }

    public function creditHistory(): HasMany
    {
        return $this->hasMany(UserCreditHistory::class);
    }

    public function memberCategories(): BelongsToMany
    {
        return $this->belongsToMany(MemberCategory::class, 'member_category_user');
    }

    public function leads(): BelongsToMany
    {
        return $this->belongsToMany(Lead::class, 'lead_user')
            ->withPivot('lead_status');
    }

    public function classifieds(): HasMany
    {
        return $this->hasMany(Classified::class);
    }

    public function jobs(): HasMany
    {
        return $this->hasMany(Job::class);
    }

    public function jobApplications(): HasMany
    {
        return $this->hasMany(JobApplication::class, 'applicant_id', 'id');
    }

    public function projects(): HasMany
    {
        return $this->hasMany(Project::class);
    }

    public function subscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class);
    }

    public function activeSubscriptions(): HasMany
    {
        return $this->hasMany(Subscription::class)->where('status', SubscriptionStatus::Active->value);
    }

    public function currentSubscription()
    {
        return $this->activeSubscriptions()->first();
    }

    public function membershipPlans(): BelongsToMany
    {
        return $this->belongsToMany(MembershipPlan::class, 'subscriptions', 'user_id', 'membership_plan_id');
    }

    public function activeMembershipPlans(): BelongsToMany
    {
        return $this->belongsToMany(MembershipPlan::class, 'subscriptions', 'user_id', 'membership_plan_id')->where('status', SubscriptionStatus::Active->value);
    }

    public function currentMembershipPlan()
    {
        return $this->activeMembershipPlans()->first();
    }

    public function team(): HasMany
    {
        return $this->hasMany(TeamMember::class);
    }

    public function articles(): HasMany
    {
        return $this->hasMany(Post::class, 'created_by');
    }

    public function services(): HasMany
    {
        return $this->hasMany(UserService::class);
    }

    public function locations(): HasMany
    {
        return $this->hasMany(UserLocation::class);
    }

    public function businessHours(): HasMany
    {
        return $this->hasMany(BusinessHour::class)
            ->orderByRaw("FIELD(day_of_week, 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday')");
    }

    public function reviews(): MorphMany
    {
        return $this->morphMany(Review::class, 'reviewable');
    }

    public function purchases(): HasMany
    {
        return $this->hasMany(\App\Models\Membership\Purchase::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(\App\Models\Membership\Payment::class);
    }

    public function videos(): HasMany
    {
        return $this->hasMany(UserVideo::class);
    }

    public function equipments(): HasMany
    {
        return $this->hasMany(Equipment::class);
    }

    public function favoriteMembers(): MorphToMany
    {
        return $this->morphedByMany(User::class, 'model', 'user_favorites', 'user_id', 'model_id')->withTimestamps();
    }
    public function favoriteClassified(): MorphToMany
    {
        return $this->morphedByMany(Classified::class, 'model', 'user_favorites', 'user_id', 'model_id')->withTimestamps();
    }

    public function favoriteJob(): MorphToMany
    {
        return $this->morphedByMany(Job::class, 'model', 'user_favorites', 'user_id', 'model_id')->withTimestamps();
    }

    public function favoriteProject(): MorphToMany
    {
        return $this->morphedByMany(Project::class, 'model', 'user_favorites', 'user_id', 'model_id')->withTimestamps();
    }

    public function favoriteArticle(): MorphToMany
    {
        return $this->morphedByMany(Post::class, 'model', 'user_favorites', 'user_id', 'model_id')->withTimestamps();
    }

    public function favoriteEquipment(): MorphToMany
    {
        return $this->morphedByMany(Equipment::class, 'model', 'user_favorites', 'user_id', 'model_id')->withTimestamps();
    }

    // Define Scopes
    public function scopeActive($query)
    {
        return $query->whereHas('profile', function ($query) {
            $query->where('status', 'active');
        });
    }


    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('avatar')
            ->singleFile()
            ->useFallbackUrl(asset('images/user_avatar_placeholder.png'))
            ->useFallbackPath(asset('images/user_avatar_placeholder.png'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('small')
                    ->fit(Fit::Crop, 150, 150)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Crop, 300, 300)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });

        $this->addMediaCollection('logo')
            ->singleFile()
            ->useFallbackUrl(asset('images/logo_placeholder.png'))
            ->useFallbackPath(asset('images/logo_placeholder.png'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('small')
                    ->fit(Fit::Contain, 150)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 300)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });

        $this->addMediaCollection('cover')
            ->singleFile()
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 1920)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });

        $this->addMediaCollection('gallery')
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 318)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 600)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });
        $this->addMediaCollection('resume')
            ->singleFile()
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 318)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });
    }

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('name')->saveSlugsTo('slug');
    }
}
