<?php

namespace App\Models\User;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Support\Str;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use App\Models\Traits\HasUuidTrait;
use Spatie\Image\Enums\Fit;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use RalphJSmit\Laravel\SEO\Support\SEOData;

class UserVideo extends Model implements HasMedia
{
    use HasFactory, SoftDeletes, InteractsWithMedia, HasUuidTrait, HasSEO;

    protected $table = 'user_videos';

    protected $fillable = [
        'uuid',
        'user_id',
        'video_title',
        'video_description',
        'video_link',
        'views',
        'likes',
        'tags',
        'is_featured',
        'is_active',
        'created_by'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'views' => 'integer',
        'likes' => 'integer',
        'tags' => 'array',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($video) {
            $video->uuid = $video->uuid ?? (string) Str::uuid();
        });
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function author()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getPermalinkAttribute(): string
    {
        return route('member.video', ['uuid' => $this->uuid]);
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('video')
            ->acceptsMimeTypes([
                'video/mp4',
                'video/webm',
                'video/ogg',
                'video/avi',
                'video/mov',
                'video/wmv',
                'video/flv'
            ])
            ->useFallbackUrl(asset('/images/image_placeholder.webp'))
            ->useFallbackPath(asset('/images/image_placeholder.webp'))
            ->singleFile()
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->format('jpg')
                    ->fit(Fit::Contain, 329)
                    ->extractVideoFrameAtSecond(5);
            });
    }

    public function getVideoUrlAttribute(): ?string
    {
        return $this->getFirstMediaUrl('video');
    }

    public function getThumbnailUrlAttribute(): ?string
    {
        return $this->getFirstMediaUrl('video', 'thumb');
    }

    public function getDynamicSEOData(): SEOData
    {
        return new SEOData(
            title: $this->video_title,
            description: $this->video_description,
            image: $this->getFirstMediaUrl('video', 'thumb'),
            url: $this->permalink,
            type: 'video',
        );
    }
}
