<?php

namespace App\Notifications;

use App\Models\Membership\Purchase;
use App\Models\Membership\Payment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;

class CreditPurchaseFailed extends Notification implements ShouldQueue
{
    use Queueable;

    protected Purchase $purchase;
    protected Payment $payment;
    protected string $errorMessage;

    /**
     * Create a new notification instance.
     */
    public function __construct(Purchase $purchase, Payment $payment, string $errorMessage = '')
    {
        $this->purchase = $purchase;
        $this->payment = $payment;
        $this->errorMessage = $errorMessage ?: 'Unknown error occurred';
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $appName = get_site_name();
        $amount = esp_currency_formated($this->purchase->total_amount, $this->purchase->currency);

        return (new MailMessage)
            ->subject(__('Credit Purchase Failed - :app', ['app' => $appName]))
            ->greeting(__('Hello :name!', ['name' => $notifiable->name]))
            ->line(__('Unfortunately, your credit purchase could not be processed.'))
            ->line(__('**Purchase Details:**'))
            ->line(__('Credits: :credits', ['credits' => $this->purchase->quantity]))
            ->line(__('Amount: :amount', ['amount' => $amount]))
            ->line(__('Error: :error', ['error' => $this->errorMessage]))
            ->line(__('**What to do next:**'))
            ->line(__('• Check your payment method details'))
            ->line(__('• Ensure you have sufficient funds'))
            ->line(__('• Try again with a different payment method'))
            ->line(__('• Contact support if the issue persists'))
            ->action(__('Try Again'), route('dashboard.credits.purchase'))
            ->line(__('If you continue to experience issues, please contact our support team.'));
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'credit_purchase_failed',
            'title' => __('Credit Purchase Failed'),
            'message' => __('Your credit purchase for :credits credits failed. :error', [
                'credits' => $this->purchase->quantity,
                'error' => $this->errorMessage
            ]),
            'purchase_id' => $this->purchase->id,
            'payment_id' => $this->payment->id,
            'amount' => $this->purchase->total_amount,
            'currency' => $this->purchase->currency,
            'error_message' => $this->errorMessage,
            'transaction_id' => $this->payment->transaction_id,
            'action_url' => route('dashboard.credits.purchase'),
            'action_text' => __('Try Again'),
        ];
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return $this->toDatabase($notifiable);
    }
}
