<?php

namespace App\Notifications;

use App\Models\Membership\Purchase;
use App\Models\Membership\Payment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Messages\DatabaseMessage;
use Illuminate\Notifications\Notification;

class CreditPurchaseSuccessful extends Notification implements ShouldQueue
{
    use Queueable;

    protected Purchase $purchase;
    protected Payment $payment;
    protected int $creditsAdded;

    /**
     * Create a new notification instance.
     */
    public function __construct(Purchase $purchase, Payment $payment, int $creditsAdded)
    {
        $this->purchase = $purchase;
        $this->payment = $payment;
        $this->creditsAdded = $creditsAdded;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $appName = get_site_name();
        $amount = esp_currency_formated($this->purchase->total_amount, $this->purchase->currency);

        return (new MailMessage)
            ->subject(__('Credit Purchase Successful - :app', ['app' => $appName]))
            ->greeting(__('Hello :name!', ['name' => $notifiable->name]))
            ->line(__('Your credit purchase has been successfully processed.'))
            ->line(__('**Purchase Details:**'))
            ->line(__('Credits Purchased: :credits', ['credits' => $this->creditsAdded]))
            ->line(__('Amount Paid: :amount', ['amount' => $amount]))
            ->line(__('Transaction ID: :id', ['id' => $this->payment->transaction_id]))
            ->line(__('Purchase Date: :date', ['date' => $this->purchase->created_at->format('M d, Y H:i')]))
            ->line(__('The credits have been added to your account and are available for use immediately.'))
            ->action(__('View Account'), route('dashboard.account-subscription'))
            ->line(__('Thank you for your purchase!'));
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'credit_purchase_successful',
            'title' => __('Credit Purchase Successful'),
            'message' => __(':credits credits have been added to your account.', ['credits' => $this->creditsAdded]),
            'purchase_id' => $this->purchase->id,
            'payment_id' => $this->payment->id,
            'amount' => $this->purchase->total_amount,
            'currency' => $this->purchase->currency,
            'credits_added' => $this->creditsAdded,
            'transaction_id' => $this->payment->transaction_id,
            'action_url' => route('dashboard.account-subscription'),
            'action_text' => __('View Account'),
        ];
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return $this->toDatabase($notifiable);
    }
}
