<?php

namespace App\Observers;

use App\Models\Review;
use App\Models\User\User;
use App\Services\Review\ReviewService;

class ReviewObserver
{
    protected ReviewService $reviewService;

    public function __construct(ReviewService $reviewService)
    {
        $this->reviewService = $reviewService;
    }

    /**
     * Handle the Review "created" event.
     */
    public function created(Review $review): void
    {
        $this->updateUserRating($review);
    }

    /**
     * Handle the Review "updated" event.
     */
    public function updated(Review $review): void
    {
        $this->updateUserRating($review);
    }

    /**
     * Handle the Review "deleted" event.
     */
    public function deleted(Review $review): void
    {
        $this->updateUserRating($review);
    }

    /**
     * Handle the Review "restored" event.
     */
    public function restored(Review $review): void
    {
        $this->updateUserRating($review);
    }

    /**
     * Handle the Review "force deleted" event.
     */
    public function forceDeleted(Review $review): void
    {
        $this->updateUserRating($review);
    }

    /**
     * Update user's profile rating when a review changes
     */
    private function updateUserRating(Review $review): void
    {
        // Only update rating if the review is for a user
        if ($review->reviewable_type === User::class && $review->reviewable_id) {
            $user = User::find($review->reviewable_id);

            if ($user && $user->profile) {
                // Calculate average rating using ReviewService
                $averageRating = $this->reviewService->getAverageRating(User::class, $user->id);

                // Update profile rating
                $user->profile->update([
                    'profile_rating' => $averageRating
                ]);
            }
        }
    }
}
