<?php

namespace App\Observers;

use App\Models\Setting;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class SettingObserver
{
    /**
     * Handle the Setting "deleting" event.
     */
    public function deleting(Setting $setting): void
    {
        // Clean up file when setting is being deleted
        if ($this->isFileType($setting->type) && !empty($setting->value)) {
            $this->cleanupFile($setting->value);
        }
    }

    /**
     * Handle the Setting "updating" event.
     */
    public function updating(Setting $setting): void
    {
        // Clean up old file when file setting is being updated
        if ($this->isFileType($setting->type)) {
            $oldValue = $setting->getOriginal('value');
            $newValue = $setting->value;

            // Only cleanup if the value is actually changing and old value exists
            if ($oldValue && $oldValue !== $newValue && !empty($oldValue)) {
                $this->cleanupFile($oldValue);
            }
        }
    }

    /**
     * Check if a type represents a file upload
     */
    protected function isFileType(?string $type): bool
    {
        return in_array($type, ['file', 'image', 'document']);
    }

    /**
     * Delete a file from storage
     */
    protected function cleanupFile(?string $filePath): void
    {
        if (!$filePath || empty($filePath)) {
            return;
        }

        try {
            // Handle both full paths and relative paths
            $relativePath = $filePath;

            // If it's a full URL, extract the relative path
            if (str_contains($filePath, '/storage/')) {
                $relativePath = str_replace('/storage/', '', parse_url($filePath, PHP_URL_PATH));
            }

            // Remove leading slash if present
            $relativePath = ltrim($relativePath, '/');

            // Use the public disk (same as Filament FileUpload default)
            if (Storage::disk('public')->exists($relativePath)) {
                Storage::disk('public')->delete($relativePath);
                Log::info("Cleaned up settings file via observer: {$relativePath}");
            }
        } catch (\Exception $e) {
            Log::error("Failed to cleanup settings file via observer: {$filePath}", [
                'error' => $e->getMessage()
            ]);
        }
    }
}
