<?php

namespace App\Observers;

use App\Models\User\UserCreditHistory;
use App\Models\User\UserProfile;

class UserCreditHistoryObserver
{
    /**
     * Handle the UserCreditHistory "created" event.
     */
    public function created(UserCreditHistory $userCreditHistory): void
    {
        $this->updateUserCreditBalance($userCreditHistory);
    }

    /**
     * Handle the UserCreditHistory "updated" event.
     */
    public function updated(UserCreditHistory $userCreditHistory): void
    {
        $this->updateUserCreditBalance($userCreditHistory);
    }

    /**
     * Handle the UserCreditHistory "deleted" event.
     */
    public function deleted(UserCreditHistory $userCreditHistory): void
    {
        $this->updateUserCreditBalance($userCreditHistory);
    }

    /**
     * Update user's credit balance based on credit history
     */
    private function updateUserCreditBalance(UserCreditHistory $userCreditHistory): void
    {
        try {
            // Calculate total credit balance from all credit history records
            $totalCredits = UserCreditHistory::where('user_id', $userCreditHistory->user_id)
                ->sum('credit');

            $totalDebits = UserCreditHistory::where('user_id', $userCreditHistory->user_id)
                ->sum('debit');

            $balance = $totalCredits - $totalDebits;

            // Update the current UserCreditHistory record with the calculated balance
            // Use updateQuietly to avoid triggering observer again
            $userCreditHistory->updateQuietly([
                'balance' => $balance
            ]);

            // Update user profile credit balance
            UserProfile::updateOrCreate(
                ['user_id' => $userCreditHistory->user_id],
                ['credit_balance' => max(0, $balance)] // Ensure balance is never negative
            );
        } catch (\Exception $e) {
            // Silently handle errors
        }
    }
}
