<?php

namespace App\Providers;

use App\Services\Settings\SettingsManager;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;

class DynamicConfigServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Only run when the app is fully booted and not during console commands
        // that might run before database is available
        if ($this->shouldSkipConfiguration()) {
            return;
        }

        // Delay configuration until after the app has fully booted
        $this->app->booted(function () {
            try {
                $this->configureDynamicSettings();
            } catch (\Exception $e) {
                // Fail silently if database is not available
                // This prevents issues during fresh installations or migrations
                logger()->warning('Failed to load dynamic configuration from database: ' . $e->getMessage());
            }
        });
    }

    /**
     * Determine if we should skip configuration
     */
    protected function shouldSkipConfiguration(): bool
    {
        return $this->app->runningInConsole() &&
            (
                $this->app->runningUnitTests() ||
                !$this->isDatabaseAvailable()
            );
    }

    /**
     * Check if database is available
     */
    protected function isDatabaseAvailable(): bool
    {
        try {
            DB::connection()->getPdo();
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Configure all dynamic settings
     */
    protected function configureDynamicSettings(): void
    {
        if (!$this->app->bound(SettingsManager::class)) {
            return;
        }

        $this->configureMail();
        $this->configureApp();
    }

    /**
     * Configure mail settings from database
     */
    protected function configureMail(): void
    {
        $settings = app(SettingsManager::class);

        // Configure SMTP settings if they exist in database
        $smtpHost = $settings->getByGroup('email', 'smtp_host');
        if ($smtpHost) {
            Config::set('mail.mailers.smtp.host', $smtpHost);
        }

        $smtpPort = $settings->getByGroup('email', 'smtp_port');
        if ($smtpPort) {
            Config::set('mail.mailers.smtp.port', (int) $smtpPort);
        }

        $smtpUsername = $settings->getByGroup('email', 'smtp_username');
        if ($smtpUsername) {
            Config::set('mail.mailers.smtp.username', $smtpUsername);
        }

        $smtpPassword = $settings->getByGroup('email', 'smtp_password');
        if ($smtpPassword) {
            Config::set('mail.mailers.smtp.password', $smtpPassword);
        }

        $smtpEncryption = $settings->getByGroup('email', 'smtp_encryption');
        if ($smtpEncryption) {
            Config::set('mail.mailers.smtp.encryption', $smtpEncryption);
        }

        // Configure from address
        $fromAddress = $settings->getByGroup('email', 'from_address');
        if ($fromAddress) {
            Config::set('mail.from.address', $fromAddress);
        }

        $fromName = $settings->getByGroup('email', 'from_name');
        if ($fromName) {
            Config::set('mail.from.name', $fromName);
        }
    }

    /**
     * Configure app settings from database
     */
    protected function configureApp(): void
    {
        $settings = app(SettingsManager::class);

        // Configure app name
        $appName = $settings->getByGroup('general', 'app_name');
        if ($appName) {
            Config::set('app.name', $appName);
        }

        // Configure timezone
        $timezone = $settings->getByGroup('general', 'app_timezone');
        if ($timezone) {
            Config::set('app.timezone', $timezone);
            if (function_exists('date_default_timezone_set')) {
                date_default_timezone_set($timezone);
            }
        }

        // Configure maintenance mode
        $maintenanceMode = $settings->getByGroup('general', 'maintenance_mode');
        if ($maintenanceMode === 'enabled') {
            // You can add custom maintenance mode logic here
            // or integrate with Laravel's maintenance mode
        }
    }
}
