<?php

namespace App\Providers;

use Illuminate\Support\Facades\Gate;
use Laravel\Telescope\IncomingEntry;
use Laravel\Telescope\Telescope;
use Laravel\Telescope\TelescopeApplicationServiceProvider;
use Laravel\Telescope\EntryType;

class TelescopeServiceProvider extends TelescopeApplicationServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {
        $this->hideSensitiveRequestDetails();

        $isLocal = $this->app->environment('local');

        Telescope::filter(function (IncomingEntry $entry) use ($isLocal) {
            if ($isLocal) {
                return true; // keep everything locally
            }

            // Always keep these in production
            if (
                $entry->isReportableException() ||
                $entry->isFailedRequest() ||    // covers many 4xx/5xx automatically
                $entry->isFailedJob() ||
                $entry->isScheduledTask() ||
                $entry->hasMonitoredTag()
            ) {
                return true;
            }

            // Requests: keep only failures (drop 2xx/3xx)
            if ($entry->type === EntryType::REQUEST) {
                $status = $entry->content['response_status'] ?? null;
                return $status && $status >= 400;
            }

            // Keep the other types you care about (but not requests)
            return in_array($entry->type, [
                EntryType::QUERY,
                EntryType::JOB,
                EntryType::LOG,
                EntryType::CACHE,
                EntryType::MODEL,
                EntryType::EVENT,
                EntryType::MAIL,
                EntryType::NOTIFICATION,
                EntryType::COMMAND,
                EntryType::REDIS,
                EntryType::VIEW,
            ], true);
        });
    }

    /**
     * Prevent sensitive request details from being logged by Telescope.
     */
    protected function hideSensitiveRequestDetails(): void
    {
        if ($this->app->environment('local')) {
            return;
        }

        Telescope::hideRequestParameters(['_token']);

        Telescope::hideRequestHeaders([
            'cookie',
            'x-csrf-token',
            'x-xsrf-token',
        ]);
    }

    /**
     * Register the Telescope gate.
     *
     * This gate determines who can access Telescope in non-local environments.
     */
    protected function gate(): void
    {
        Gate::define('viewTelescope', function ($user) {

            if ($this->app->environment('local')) {
                return true;
            }

            return in_array($user->email, [
                'shanijahania@gmail.com',
                'admin@espmarketplace.com'
            ]);
        });
    }
}
