<?php

declare(strict_types=1);

namespace App\Repositories\Contracts\Credit;

use Illuminate\Database\Eloquent\Model;

interface CreditRepositoryInterface
{
    /**
     * Get the base price per credit
     */
    public function getCreditPrice(): float;

    /**
     * Get the credit currency
     */
    public function getCreditCurrency(): string;

    /**
     * Check if bulk discounts are enabled
     */
    public function areBulkDiscountsEnabled(): bool;

    /**
     * Get all bulk discount tiers
     */
    public function getBulkDiscountTiers(): array;

    /**
     * Calculate the best discount percentage for a given quantity
     */
    public function calculateDiscountPercentage(int $quantity): float;

    /**
     * Calculate the total price for a given quantity of credits
     */
    public function calculateTotalPrice(int $quantity): array;

    /**
     * Get the minimum credits per purchase limit
     */
    public function getMinCreditsPerPurchase(): int;

    /**
     * Get the maximum credits per purchase limit
     */
    public function getMaxCreditsPerPurchase(): int;

    /**
     * Check if credit expiry is enabled
     */
    public function isCreditExpiryEnabled(): bool;

    /**
     * Get credit expiry period in days
     */
    public function getCreditExpiryDays(): int;

    /**
     * Check if credit refunds are enabled
     */
    public function areCreditRefundsEnabled(): bool;

    /**
     * Check if credit transfers are enabled
     */
    public function areCreditTransfersEnabled(): bool;

    /**
     * Validate quantity against limits
     */
    public function validateQuantityLimits(int $quantity): void;

    /**
     * Get pricing tiers for display
     */
    public function getPricingTiers(): array;

    /**
     * Format currency amount for display
     */
    public function formatCurrency(float $amount): string;

    // Merged methods from UserCreditHistoryRepositoryInterface
    /**
     * Get all credit history records with optional filters
     */
    public function getAll(array $args = []);

    /**
     * Find credit history record by ID
     */
    public function findById(int $id): ?Model;

    /**
     * Find credit history record by UUID
     */
    public function findByUuid(string $uuid): ?Model;

    /**
     * Create a new credit history record
     */
    public function create(array $data): Model;

    /**
     * Update an existing credit history record
     */
    public function update(Model $model, array $data): bool;

    /**
     * Update credit history record by UUID
     */
    public function updateByUuid(string $uuid, array $data): bool;

    /**
     * Delete a credit history record
     */
    public function delete(Model $model): bool;

    /**
     * Delete credit history record by ID
     */
    public function deleteById(int $id): bool;

    /**
     * Delete credit history record by UUID
     */
    public function deleteByUuid(string $uuid): bool;
}
