<?php

namespace App\Repositories\Eloquent\ApiClient;

use App\Models\ApiClient;
use App\Repositories\Contracts\ApiClient\ApiClientRepositoryInterface;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;

class ApiClientRepository implements ApiClientRepositoryInterface
{
    public function getAll(array $filters = [], array $with = []): LengthAwarePaginator
    {
        $query = ApiClient::query();

        // Apply relationships
        if (!empty($with)) {
            $query->with($with);
        }

        // Apply filters
        foreach ($filters as $key => $value) {
            if ($key === 'type' && $value) {
                $query->where('type', $value);
            } elseif ($key === 'is_active' && $value !== null) {
                $query->where('is_active', $value);
            } elseif ($key === 'created_by' && $value) {
                $query->where('created_by', $value);
            } elseif ($key === 'search' && $value) {
                $query->where('name', 'like', '%' . $value . '%');
            } elseif ($key === 'order_by' && $value) {
                $orderDirection = $filters['order_direction'] ?? 'desc';
                $query->orderBy($value, $orderDirection);
            }
        }

        // Default ordering
        if (!isset($filters['order_by'])) {
            $query->orderBy('created_at', 'desc');
        }

        $perPage = $filters['per_page'] ?? 15;
        return $query->paginate($perPage);
    }

    public function getAllActive(array $with = []): Collection
    {
        $query = ApiClient::where('is_active', true);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->get();
    }

    public function findById(int $id, array $with = []): ?ApiClient
    {
        $query = ApiClient::where('id', $id);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->first();
    }

    public function findByIdOrFail(int $id, array $with = []): ApiClient
    {
        $query = ApiClient::where('id', $id);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->firstOrFail();
    }

    public function create(array $data): ApiClient
    {
        return ApiClient::create($data);
    }

    public function update(ApiClient $apiClient, array $data): ApiClient
    {
        $apiClient->update($data);
        return $apiClient->fresh();
    }

    public function delete(ApiClient $apiClient): bool
    {
        return $apiClient->delete();
    }

    public function activate(ApiClient $apiClient): ApiClient
    {
        $apiClient->update(['is_active' => true]);
        return $apiClient->fresh();
    }

    public function deactivate(ApiClient $apiClient): ApiClient
    {
        $apiClient->update(['is_active' => false]);
        return $apiClient->fresh();
    }

    public function revokeAllTokens(ApiClient $apiClient): void
    {
        $apiClient->tokens()->delete();
    }
}
