<?php

namespace App\Repositories\Eloquent\Blog;

use App\Models\Blog\Post;
use App\Repositories\Contracts\Blog\PostRepositoryInterface;

class PostRepository implements PostRepositoryInterface
{
    public function getAll($args = [])
    {
        $query = Post::query();
        $str   = $args['search'] ?? null;
        if (! empty($str)) {
            $query->where(function ($q) use ($str) {
                $q->where('post_title', 'like', '%' . $str . '%');
            });
        }
        if (! empty($args['category'])) {
            dump($args['category']);
            if (is_array($args['category'])) {
                if (!in_array('all', $args['category'])) {
                    $query->whereHas('categories', function ($q) use ($args) {
                        $q->whereIn('slug', $args['category']);
                    });
                }
            } else {
                $query->whereHas('categories', function ($q) use ($args) {
                    $q->where('slug', $args['category']);
                });
            }
        }

        if (! empty($args['tag'])) {
            $query->whereHas('tags', function ($q) use ($args) {
                $q->where('slug', $args['tag']);
            });
        }

        if (! empty($args['author'])) {
            $query->whereHas('author', function ($q) use ($args) {
                $q->where('slug', $args['author']);
            });
        } else {
            $query->when(! empty($args['created_by']), function ($q) use ($args) {
                $q->where('created_by', $args['created_by']);
            });
        }
        // type
        if (! empty($args['type'])) {
            switch ($args['type']) {
                case 'feautred-first':
                    $query->featured();
                    break;
                case 'latest-first':
                    $query->latest();
                    break;
                case 'most-popular':
                    $query->popular();
                    break;
            }
        }
        // orderBy and order
        if (! empty($args['order_by'])) {
            switch ($args['order_by']) {
                case 'is_featured':
                    $query->where('is_featured', 1)->orderBy('is_featured', $args['order']);
                    break;
                case 'published_at':
                    $query->latest();
                    break;
                case 'updated_at':
                    $query->popular();
                    break;
                default:
                    $query->orderBy($args['order_by'], $args['order']);
                    break;
            }
        }
        return $query;
    }

    public function getScheduledForPublishing()
    {
        return Post::scheduledForPublishing();
    }

    public function findBySlug($slug)
    {
        return Post::where('slug', $slug)->first();
    }

    public function findByUuid(string $uuid)
    {
        return Post::where('uuid', $uuid)->first();
    }

    public function find($id)
    {
        return Post::find($id);
    }

    public function create(array $data)
    {
        return Post::create($data);
    }

    public function update($id, array $data)
    {
        return Post::find($id)->update($data);
    }

    public function updateByUuid(string $uuid, array $data)
    {
        $post = Post::where('uuid', $uuid)->firstOrFail();
        $post->update($data);
        return $post;
    }

    public function delete($id)
    {
        return Post::find($id)->delete();
    }

    public function getRelatedPosts($args = [])
    {
        $post       = $args['post'];
        $limit      = $args['limit'] ?? 3;
        $categories = $post->categories->pluck('id');
        return Post::published()
            ->where('posts.id', '!=', $post->id)
            ->whereHas('categories', function ($q) use ($categories) {
                $q->whereIn('post_categories.id', $categories);
            })
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }

    public function getFeaturedPosts($args = [])
    {
        $limit = $args['limit'] ?? 3;
        return Post::featured()
            ->published()
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }
}
