<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\Classified;

use App\Models\Classified\Classified;
use App\Models\Location\Country;
use App\Models\Location\City;
use App\Models\Location\State;
use App\Repositories\Contracts\Classified\ClassifiedRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class ClassifiedRepository implements ClassifiedRepositoryInterface
{
    public function getAll(array $args = []): Builder
    {
        $query = Classified::query();

        // dump($args);
        if (!empty($args['search'])) {
            $query->where(function ($q) use ($args) {
                $q->where('title', 'like', '%' . $args['search'] . '%')
                    ->orWhere('description', 'like', '%' . $args['search'] . '%');
            });
        }

        if (!empty($args['flashOnly'])) {
            $query->onlyFlashSale();
        }

        if (!empty($args['category'])) {
            $query->whereHas('categories', function ($q) use ($args) {
                $q->where('slug', $args['category']);
            });
        }

        if (!empty($args['user_id'])) {
            $query->where('user_id', $args['user_id']);
        }

        // Exclude current classified if provided
        if (!empty($args['exclude_id'])) {
            $query->where('id', '!=', $args['exclude_id']);
        }

        // Exclude current classified by UUID if provided
        if (!empty($args['exclude_uuid'])) {
            $query->where('uuid', '!=', $args['exclude_uuid']);
        }

        if (!empty($args['limit'])) {
            $query->limit($args['limit']);
        }
        if (!empty($args['classified_status'])) {
            $classifiedStatus = (array) $args['classified_status'];
            $query->whereIn('classified_status', $classifiedStatus);
        }
        // Location based filtering
        if (!empty($args['radius']) && !empty($args['lat']) && !empty($args['lng'])) {
            // Use radius search with the provided coordinates
            $radius = (float) $args['radius'];
            $lat = (float) $args['lat'];
            $lng = (float) $args['lng'];

            // Add distance calculation and filter by radius using Haversine formula
            $query->selectRaw('classifieds.*, (
                6371 * acos(
                    cos(radians(?)) * 
                    cos(radians(latitude)) * 
                    cos(radians(longitude) - radians(?)) + 
                    sin(radians(?)) * 
                    sin(radians(latitude))
                )
            ) AS distance', [$lat, $lng, $lat])
                ->whereNotNull('latitude')
                ->whereNotNull('longitude')
                ->havingRaw('distance <= ?', [$radius])
                ->orderBy('distance', 'asc');
        } else {
            // Fallback to city/state/country filtering
            if (!empty($args['city'])) {
                $query->where('city_id', $args['city']);
            } elseif (!empty($args['state'])) {
                $query->where('state_id', $args['state']);
            } elseif (!empty($args['country'])) {
                $query->where('country_id', $args['country']);
            }
        }

        // Add type filter
        if (!empty($args['type'])) {
            $query->whereHas('types', function ($q) use ($args) {
                $typeIds = (array) $args['type'];
                $q->whereIn('classified_types.id', $typeIds);
            });
        }

        // Add main category filter
        if (!empty($args['categories'])) {
            $query->whereHas('categories', function ($q) use ($args) {
                $categoryIds = (array) $args['categories'];
                $q->whereIn('classified_categories.id', $categoryIds);
            });
        }

        // Add condition filter
        if (!empty($args['condition'])) {
            $condition = (array)$args['condition'];
            $query->whereIn('condition', $condition);
        }

        // Add seller type filter
        if (!empty($args['sellerType']) && is_array($args['sellerType'])) {
            $query->whereHas('author', function ($q) use ($args) {
                $sellerType = (array) $args['sellerType'];
                $q->whereHas('profile', function ($q) use ($sellerType) {
                    $q->whereIn('profile_type', $sellerType);
                });
            });
        }

        // Add delivery filter
        if (!empty($args['deliveries'])) {
            $query->whereHas('deliveries', function ($q) use ($args) {
                $deliveryIds = (array) $args['deliveries'];
                $q->whereIn('classified_deliveries.id', $deliveryIds);
            });
        }

        // Add tag filter
        if (!empty($args['tag'])) {
            $query->whereHas('tags', function ($q) use ($args) {
                $q->where('slug', $args['tag']);
            });
        }

        // Add item age filter
        if (!empty($args['itemAge'])) {
            // dump($args['itemAge']);
            if ($args['itemAge']['from'] !== null && $args['itemAge']['to'] !== null) {
                $query->whereBetween('manufacture_date', [
                    now()->subDays($args['itemAge']['to']),
                    now()->subDays($args['itemAge']['from'])
                ]);
            } elseif ($args['itemAge']['from'] !== null) {
                $query->where('manufacture_date', '<=', now()->subDays($args['itemAge']['from']));
            } elseif ($args['itemAge']['to'] !== null) {
                $query->where('manufacture_date', '>=', now()->subDays($args['itemAge']['to']));
            }
        }

        // Add Date Posted filter
        if (!empty($args['datePosted'])) {
            $query->where('created_at', '>=', now()->subDays($args['datePosted']));
        }

        // Add ordering (skip if already ordered by distance)
        if (!empty($args['radius']) && !empty($args['lat']) && !empty($args['lng'])) {
            // Distance ordering is already applied above
        } elseif (!empty($args['order_by'])) {
            $order = !empty($args['order']) && in_array(strtolower($args['order']), ['asc', 'desc'])
                ? strtolower($args['order'])
                : 'asc';

            $query->orderBy($args['order_by'], $order);
        } else {
            // Default ordering if not specified
            $query->orderBy('created_at', 'desc');
        }


        return $query;
    }

    public function getScheduledForPublishing(): Builder
    {
        return Classified::scheduledForPublishing();
    }

    public function find(int $id): ?Model
    {
        return Classified::find($id);
    }

    public function findBySlug(string $slug): ?Model
    {
        return Classified::where('slug', $slug)->first();
    }

    public function create(array $data): Model
    {
        // dd($data);
        $classified = Classified::create($data);
        // dd($classified);
        $this->saveLocation($classified, $data);
        return $classified;
    }

    public function update(int $id, array $data): bool
    {
        return Classified::find($id)->update($data);
    }

    public function delete(int $id): bool
    {
        return Classified::find($id)->delete();
    }
    public function findByUuid(string $uuid)
    {
        return Classified::where('uuid', $uuid)->first();
    }
    public function updateByUuid(string $uuid, array $data)
    {
        $classified = Classified::where('uuid', $uuid)->firstOrFail();
        $classified->update($data);
        $this->saveLocation($classified, $data);
        return $classified;
    }

    public function saveLocation(Classified $classified, array $data)
    {
        $countryName = !empty($data['country_name']) ? $data['country_name'] : null;
        $countryCode = !empty($data['country_code']) ? $data['country_code'] : null;
        $cityName = !empty($data['city_name']) ? $data['city_name'] : null;
        $stateName = !empty($data['state_name']) ? $data['state_name'] : null;
        $stateCode = !empty($data['state_code']) ? $data['state_code'] : null;
        $location = !empty($data['location']) ? $data['location'] : null;
        $address = !empty($data['address']) ? $data['address'] : null;

        $classified->location = $location;

        // Create country if not exists
        if ($countryName) {

            $country = Country::firstOrCreate(
                ['country_name' => $countryName],
                ['country_code' => $countryCode]
            );
            $classified->country_id = $country->id;

            // Create state if not exists
            if ($stateName) {
                $state = State::firstOrCreate(
                    ['country_id' => $country->id, 'state_name' => $stateName],
                    ['state_code' => $stateCode]
                );
                $classified->state_id = $state->id;
            }

            // Create city if not exists
            if ($cityName) {
                $city = City::firstOrCreate(
                    ['country_id' => $country->id, 'city_name' => $cityName],
                    ['state_id' => $state->id]
                );
                $classified->city_id = $city->id;
            }
        }

        $classified->save();
        return $classified;
    }
}
