<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\Credit;

use App\Repositories\Contracts\Credit\CreditRepositoryInterface;
use App\Services\Settings\SettingsManager;
use App\Models\User\UserCreditHistory;
use Illuminate\Database\Eloquent\Model;
use InvalidArgumentException;

class CreditRepository implements CreditRepositoryInterface
{
    public function __construct(
        private SettingsManager $settingsManager
    ) {}

    /**
     * Get the base price per credit
     */
    public function getCreditPrice(): float
    {
        return (float) $this->settingsManager->get('membership', 'credit_price', 1.00);
    }

    /**
     * Get the credit currency
     */
    public function getCreditCurrency(): string
    {
        return $this->settingsManager->get('membership', 'credit_currency', 'USD');
    }

    /**
     * Check if bulk discounts are enabled
     */
    public function areBulkDiscountsEnabled(): bool
    {
        return (bool) $this->settingsManager->get('membership', 'bulk_discounts_enabled', true);
    }

    /**
     * Get all bulk discount tiers
     */
    public function getBulkDiscountTiers(): array
    {
        $discounts = $this->settingsManager->get('membership', 'credit_bulk_discounts', '[]');

        if (is_string($discounts)) {
            $discounts = json_decode($discounts, true) ?? [];
        }

        // Sort by quantity ascending
        usort($discounts, fn($a, $b) => $a['quantity'] <=> $b['quantity']);

        return $discounts;
    }

    /**
     * Calculate the best discount percentage for a given quantity
     */
    public function calculateDiscountPercentage(int $quantity): float
    {
        if (!$this->areBulkDiscountsEnabled() || $quantity <= 0) {
            return 0.0;
        }

        $tiers = $this->getBulkDiscountTiers();
        $bestDiscount = 0.0;

        foreach ($tiers as $tier) {
            if ($quantity >= $tier['quantity']) {
                $bestDiscount = max($bestDiscount, (float) $tier['discount']);
            }
        }

        return $bestDiscount;
    }

    /**
     * Calculate the total price for a given quantity of credits
     */
    public function calculateTotalPrice(int $quantity): array
    {
        if ($quantity <= 0) {
            throw new InvalidArgumentException('Quantity must be greater than 0');
        }

        $this->validateQuantityLimits($quantity);

        $basePrice = $this->getCreditPrice();
        $subtotal = $basePrice * $quantity;
        $discountPercentage = $this->calculateDiscountPercentage($quantity);
        $discountAmount = $subtotal * ($discountPercentage / 100);
        $total = $subtotal - $discountAmount;

        return [
            'quantity' => $quantity,
            'base_price_per_credit' => $basePrice,
            'subtotal' => round($subtotal, 2),
            'discount_percentage' => $discountPercentage,
            'discount_amount' => round($discountAmount, 2),
            'total' => round($total, 2),
            'currency' => $this->getCreditCurrency(),
            'savings' => round($discountAmount, 2),
        ];
    }

    /**
     * Get the minimum credits per purchase limit
     */
    public function getMinCreditsPerPurchase(): int
    {
        return (int) $this->settingsManager->get('membership', 'min_credits_per_purchase', 1);
    }

    /**
     * Get the maximum credits per purchase limit
     */
    public function getMaxCreditsPerPurchase(): int
    {
        return (int) $this->settingsManager->get('membership', 'max_credits_per_purchase', 1000);
    }

    /**
     * Check if credit expiry is enabled
     */
    public function isCreditExpiryEnabled(): bool
    {
        return (bool) $this->settingsManager->get('membership', 'credit_expiry_enabled', false);
    }

    /**
     * Get credit expiry period in days
     */
    public function getCreditExpiryDays(): int
    {
        return (int) $this->settingsManager->get('membership', 'credit_expiry_days', 365);
    }

    /**
     * Check if credit refunds are enabled
     */
    public function areCreditRefundsEnabled(): bool
    {
        return (bool) $this->settingsManager->get('membership', 'credit_refund_enabled', false);
    }

    /**
     * Check if credit transfers are enabled
     */
    public function areCreditTransfersEnabled(): bool
    {
        return (bool) $this->settingsManager->get('membership', 'credit_transfer_enabled', false);
    }

    /**
     * Validate quantity against limits
     */
    public function validateQuantityLimits(int $quantity): void
    {
        $min = $this->getMinCreditsPerPurchase();
        $max = $this->getMaxCreditsPerPurchase();

        if ($quantity < $min) {
            throw new InvalidArgumentException("Minimum purchase quantity is {$min} credits");
        }

        if ($quantity > $max) {
            throw new InvalidArgumentException("Maximum purchase quantity is {$max} credits");
        }
    }

    /**
     * Get pricing tiers for display (useful for frontend)
     */
    public function getPricingTiers(): array
    {
        $tiers = [];
        $basePrice = $this->getCreditPrice();
        $currency = $this->getCreditCurrency();

        // Add base tier (no discount)
        $tiers[] = [
            'quantity' => 1,
            'price_per_credit' => $basePrice,
            'total_price' => $basePrice,
            'discount_percentage' => 0,
            'savings' => 0,
            'currency' => $currency,
        ];

        if ($this->areBulkDiscountsEnabled()) {
            $discountTiers = $this->getBulkDiscountTiers();

            foreach ($discountTiers as $tier) {
                $pricing = $this->calculateTotalPrice($tier['quantity']);
                $tiers[] = [
                    'quantity' => $tier['quantity'],
                    'price_per_credit' => round($pricing['total'] / $tier['quantity'], 2),
                    'total_price' => $pricing['total'],
                    'discount_percentage' => $pricing['discount_percentage'],
                    'savings' => $pricing['savings'],
                    'currency' => $currency,
                ];
            }
        }

        return $tiers;
    }

    /**
     * Format currency amount for display
     */
    public function formatCurrency(float $amount): string
    {
        $currency = $this->getCreditCurrency();

        return match ($currency) {
            'USD' => '$' . number_format($amount, 2),
            'EUR' => '€' . number_format($amount, 2),
            'GBP' => '£' . number_format($amount, 2),
            'PKR' => 'Rs ' . number_format($amount, 2),
            default => $currency . ' ' . number_format($amount, 2),
        };
    }

    // Merged methods from UserCreditHistoryRepository
    /**
     * Get all credit history records with optional filters
     */
    public function getAll(array $args = [])
    {
        $query = UserCreditHistory::query();
        if (!empty($args['user_id'])) {
            $query->where('user_id', $args['user_id']);
        }

        return $query;
    }

    /**
     * Find credit history record by ID
     */
    public function findById(int $id): ?Model
    {
        return UserCreditHistory::find($id);
    }

    /**
     * Find credit history record by UUID
     */
    public function findByUuid(string $uuid): ?Model
    {
        return UserCreditHistory::where('uuid', $uuid)->first();
    }

    /**
     * Create a new credit history record
     */
    public function create(array $data): Model
    {
        return UserCreditHistory::create($data);
    }

    /**
     * Update an existing credit history record
     */
    public function update(Model $model, array $data): bool
    {
        return $model->update($data);
    }

    /**
     * Update credit history record by UUID
     */
    public function updateByUuid(string $uuid, array $data): bool
    {
        return $this->findByUuid($uuid)->update($data);
    }

    /**
     * Delete a credit history record
     */
    public function delete(Model $model): bool
    {
        return $model->delete();
    }

    /**
     * Delete credit history record by ID
     */
    public function deleteById(int $id): bool
    {
        return $this->findById($id)->delete();
    }

    /**
     * Delete credit history record by UUID
     */
    public function deleteByUuid(string $uuid): bool
    {
        return $this->findByUuid($uuid)->delete();
    }
}
