<?php

namespace App\Repositories\Eloquent\Equipment;

use App\Models\Equipment\Equipment;
use App\Repositories\Contracts\Equipment\EquipmentRepositoryInterface;
use App\Enums\EquipmentStatus;

class EquipmentRepository implements EquipmentRepositoryInterface
{
    public function getAll($args = [])
    {
        $query = Equipment::query();
        
        if (!empty($args['search'])) {
            $query->where(function($q) use ($args) {
                $q->where('title', 'like', '%' . $args['search'] . '%')
                  ->orWhere('description', 'like', '%' . $args['search'] . '%');
            });
        }

        if (!empty($args['category'])) {
            $query->whereHas('categories', function($q) use ($args) {
                $q->where('slug', $args['category']);
            });
        }

        if (!empty($args['location'])) {
            $query->where(function($q) use ($args) {
                $q->whereHas('country', function($sq) use ($args) {
                    $sq->where('name', 'like', '%' . $args['location'] . '%');
                })->orWhereHas('state', function($sq) use ($args) {
                    $sq->where('name', 'like', '%' . $args['location'] . '%');
                })->orWhereHas('city', function($sq) use ($args) {
                    $sq->where('name', 'like', '%' . $args['location'] . '%');
                });
            });
        }

        if (!empty($args['company'])) {
            $query->whereHas('user', function($q) use ($args) {
                $q->where('company_name', 'like', '%' . $args['company'] . '%');
            });
        }

        if (!empty($args['status'])) {
            $query->where('status', $args['status']);
        }

        if (!empty($args['created_by'])) {
            $query->where('created_by', $args['created_by']);
        }

        if (!empty($args['user_id'])) {
            $query->where('user_id', $args['user_id']);
        }

        // Add tag filtering
        if (!empty($args['tags'])) {
            $tags = is_array($args['tags']) ? $args['tags'] : [$args['tags']];
            foreach ($tags as $tag) {
                $query->whereJsonContains('tags', $tag);
            }
        }

        // Add ordering
        if (!empty($args['order_by'])) {
            $order = !empty($args['order']) && in_array(strtolower($args['order']), ['asc', 'desc']) 
                ? strtolower($args['order']) 
                : 'asc';
            
            $query->orderBy($args['order_by'], $order);
        } else {
            // Default ordering if not specified
            $query->orderBy('created_at', 'desc');
        }
        
        return $query;
    }

    public function find($id)
    {
        return Equipment::find($id);
    }

    public function findBySlug($slug)
    {
        return Equipment::where('slug', $slug)->first();
    }

    public function findByUuid(string $uuid)
    {
        return Equipment::where('uuid', $uuid)->first();
    }

    public function create(array $data)
    {
        return Equipment::create($data);
    }

    public function update($id, array $data)
    {
        $equipment = Equipment::find($id);
        $equipment->update($data);
        return $equipment;
    }

    public function updateByUuid(string $uuid, array $data)
    {
        $equipment = Equipment::where('uuid', $uuid)->firstOrFail();
        $equipment->update($data);
        return $equipment;
    }

    public function delete($id)
    {
        return Equipment::find($id)->delete();
    }

    public function getFeaturedEquipment($args = [])
    {
        $limit = $args['limit'] ?? 6;
        return Equipment::where('status', EquipmentStatus::Published)
            ->when(!empty($args['exclude_id']), function($q) use ($args) {
                $q->where('id', '!=', $args['exclude_id']);
            })
            ->latest()
            ->limit($limit)
            ->get();
    }

    public function getRelatedEquipment($args = [])
    {
        $equipment = $args['equipment'];
        $limit = $args['limit'] ?? 3;
        
        return Equipment::where('equipments.id', '!=', $equipment->id)
            ->where('status', EquipmentStatus::Published)
            ->where(function($q) use ($equipment) {
                // If equipment has categories, find related by same category
                if ($equipment->categories->isNotEmpty()) {
                    $categoryIds = $equipment->categories->pluck('id')->toArray();
                    $q->whereHas('categories', function($sq) use ($categoryIds) {
                        $sq->whereIn('equipment_categories.id', $categoryIds);
                    });
                }
                
                // Or by similar tags if available
                if (!empty($equipment->tags)) {
                    foreach ($equipment->tags as $tag) {
                        $q->orWhereJsonContains('tags', $tag);
                    }
                }
            })
            ->latest()
            ->limit($limit)
            ->get();
    }
} 