<?php

namespace App\Repositories\Eloquent\Job;

use App\Enums\JobStatus;
use App\Models\Job\Job;
use App\Repositories\Contracts\Job\JobRepositoryInterface;

class JobRepository implements JobRepositoryInterface
{
    public function getAll($args = [])
    {
        $query = Job::query();

        if (! empty($args['search'])) {
            $query->where(function ($q) use ($args) {
                $q->where('job_title', 'like', '%' . $args['search'] . '%')
                    ->orWhere('job_description', 'like', '%' . $args['search'] . '%');
            });
        }

        if (!empty($args['job_role'])) {
            $jobRoles = (array) $args['job_role'];
            $ids = [];
            $slugs = [];
            foreach ($jobRoles as $jobRole) {
                if (is_numeric($jobRole)) {
                    $ids[] = (int) $jobRole;
                } else {
                    $slugs[] = $jobRole;
                }
            }
            if(!empty($ids)) {
                $query->whereIn('job_role_id', $ids);
            }

            if(!empty($slugs)) {
                $query->whereHas('job_role', function ($q) use ($slugs) {
                    $q->whereIn('slug', $slugs);
                });
            }
            // $query->whereHas('job_role', function ($q) use ($jobRoles) {
            //     $q->whereIn('slug', $jobRoles);
            // });
        }

        if (!empty($args['career_level'])) {
            $careerLevels = (array)$args['career_level'];
            $query->whereHas('job_career_level', function ($q) use ($careerLevels) {
                $q->whereIn('slug', $careerLevels);
            });
        }

        if (!empty($args['job_type'])) {
            $jobTypes = (array)$args['job_type'];
            $query->whereHas('job_type', function ($q) use ($args) {
                $q->whereIn('slug', $args['job_type']);
            });
        }


        if (! empty($args['job_industry'])) {
            $query->whereHas('job_industry', function ($q) use ($args) {
                $q->where('slug', $args['job_industry']);
            });
        }

        if (!empty($args['city']) || !empty($args['state']) || !empty($args['country'])) {
            if (!empty($args['city'])) {
                $query->where('city_id', $args['city']);
            } else if (!empty($args['state'])) {
                $query->where('state_id', $args['state']);
            } else if (!empty($args['country'])) {
                $query->where('country_id', $args['country']);
            }
        }

        if (!empty($args['shift'])) {
            $shifts = (array)$args['shift'];
            $query->whereIn('job_shift', $shifts);
        }

        if (!empty($args['work_mode'])) {
            $workModes = (array)$args['work_mode'];
            $query->whereIn('work_mode', $workModes);
        }

        if (! empty($args['company'])) {
            $query->whereHas('user', function ($q) use ($args) {
                $q->where('company_name', 'like', '%' . $args['company'] . '%');
            });
        }

        if (!empty($args['date_posted'])) {
            $query->where('created_at', '>=', now()->subDays($args['date_posted']));
        }

        if (!empty($args['open_jobs_only'])) {
            $query->where('apply_by_date', '>=', now());
        }

        if (isset($args['is_featured'])) {
            $query->where('is_featured', $args['is_featured']);
        }

        if (! empty($args['job_status'])) {
            $query->where('job_status', $args['job_status']);
        }

        if (! empty($args['created_by'])) {
            $query->where('created_by', $args['created_by']);
        }

        if (! empty($args['user_id'])) {
            $query->where('user_id', $args['user_id']);
        }
        if (!empty($args['limit'])) {
            $query->limit($args['limit']);
        }

        // Salary filtering logic
        if (!empty($args['min_salary']) && !empty($args['max_salary'])) {
            // Both min and max salary filters provided
            $query->where(function ($q) use ($args) {
                $q->where('min_salary', '>=', $args['min_salary'])
                    ->where('max_salary', '<=', $args['max_salary']);
            });
        } elseif (!empty($args['min_salary'])) {
            // Only min salary filter provided
            $query->where('min_salary', '>=', $args['min_salary']);
        } elseif (!empty($args['max_salary'])) {
            // Only max salary filter provided
            $query->where('max_salary', '<=', $args['max_salary']);
        }

        if (! empty($args['open_job'])) {
            $query->where('apply_by_date', '>=', now());
        }

        // Add ordering
        if (!empty($args['order_by'])) {
            $order = ! empty($args['order']) && in_array(strtolower($args['order']), ['asc', 'desc'])
                ? strtolower($args['order'])
                : 'asc';

            $query->orderBy($args['order_by'], $order);
        } else {
            // Default ordering if not specified
            $query->orderBy('created_at', 'desc');
        }

        return $query;
    }

    public function find($id)
    {
        return Job::find($id);
    }

    public function findBySlug($slug)
    {
        return Job::where('slug', $slug)->first();
    }

    public function findByUuid(string $uuid)
    {
        return Job::where('uuid', $uuid)->first();
    }

    public function create(array $data)
    {
        return Job::create($data);
    }

    public function update($id, array $data)
    {
        $job = Job::find($id);
        $job->update($data);
        return $job;
    }

    public function updateByUuid(string $uuid, array $data)
    {
        $job = Job::where('uuid', $uuid)->firstOrFail();
        $job->update($data);
        return $job;
    }

    public function delete($id)
    {
        return Job::find($id)->delete();
    }

    public function getFeaturedJobs($args = [])
    {
        $limit = $args['limit'] ?? 6;
        return Job::where('is_featured', true)
            ->where('job_status', JobStatus::Published)
            ->when(! empty($args['exclude_id']), function ($q) use ($args) {
                $q->where('id', '!=', $args['exclude_id']);
            })
            ->latest()
            ->limit($limit)
            ->get();
    }

    public function getRelatedJobs($args = [])
    {
        $job   = $args['job'];
        $limit = $args['limit'] ?? 3;
        return Job::where('jobs.id', '!=', $job->id)
            ->where('job_status', JobStatus::Published)
            ->whereHas('user.profile')
            ->where(function ($q) use ($job) {
                $q->where('job_role_id', $job->job_role_id)
                    ->orWhere('job_industry_id', $job->job_industry_id);
            })
            ->latest()
            ->limit($limit)
            ->get();
    }
}
