<?php

namespace App\Repositories\Eloquent\Job;

use App\Models\Job\JobSkill;
use App\Models\Job\Job;
use App\Repositories\Contracts\Job\JobSkillRepositoryInterface;

class JobSkillRepository implements JobSkillRepositoryInterface
{
    public function getAll($args = [])
    {
        $query = JobSkill::query();
        
        if (!empty($args['search'])) {
            $query->where(function($q) use ($args) {
                $q->where('job_skill_name', 'like', '%' . $args['search'] . '%')
                  ->orWhere('job_skill_description', 'like', '%' . $args['search'] . '%');
            });
        }

        if (isset($args['is_active'])) {
            $query->where('is_active', $args['is_active']);
        }

        if (!empty($args['created_by'])) {
            $query->where('created_by', $args['created_by']);
        }

        return $query;
    }

    public function find($id)
    {
        return JobSkill::find($id);
    }

    public function findBySlug($slug)
    {
        return JobSkill::where('slug', $slug)->first();
    }

    public function findByUuid(string $uuid)
    {
        return JobSkill::where('uuid', $uuid)->first();
    }

    public function create(array $data)
    {
        return JobSkill::create($data);
    }

    public function update($id, array $data)
    {
        $skill = JobSkill::find($id);
        $skill->update($data);
        return $skill;
    }

    public function updateByUuid(string $uuid, array $data)
    {
        $skill = JobSkill::where('uuid', $uuid)->firstOrFail();
        $skill->update($data);
        return $skill;
    }

    public function delete($id)
    {
        return JobSkill::find($id)->delete();
    }

    public function getActive()
    {
        return JobSkill::where('is_active', true)->get();
    }

    public function getPopularSkills($limit = 10)
    {
        return JobSkill::withCount('jobs')
            ->where('is_active', true)
            ->orderBy('jobs_count', 'desc')
            ->limit($limit)
            ->get();
    }

    public function attachToJob($jobId, array $skillIds)
    {
        $job = Job::findOrFail($jobId);
        $job->job_skills()->sync($skillIds);
        return $job->job_skills;
    }

    public function detachFromJob($jobId, array $skillIds)
    {
        $job = Job::findOrFail($jobId);
        $job->job_skills()->detach($skillIds);
        return $job->job_skills;
    }
} 