<?php

namespace App\Repositories\Eloquent\Lead;

use App\Models\Lead\LeadForm;
use App\Repositories\Contracts\Lead\LeadFormRepositoryInterface;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;

class LeadFormRepository implements LeadFormRepositoryInterface
{
    public function getAll(array $filters = [], array $with = [], array $withCount = []): LengthAwarePaginator
    {
        $query = LeadForm::query();

        // Apply relationships
        if (!empty($with)) {
            $query->with($with);
        }

        // Apply counts
        if (!empty($withCount)) {
            $query->withCount($withCount);
        }

        // Apply filters (if provided by service)
        foreach ($filters as $key => $value) {
            if ($key === 'search' && $value) {
                $query->where(function ($q) use ($value) {
                    $q->where('title', 'like', '%' . $value . '%')
                      ->orWhere('description', 'like', '%' . $value . '%');
                });
            } elseif ($key === 'is_active' && $value !== null) {
                $query->where('is_active', $value);
            } elseif ($key === 'match_criteria' && $value) {
                $query->where('match_criteria', $value);
            } elseif ($key === 'reception_mode' && $value) {
                $query->where('reception_mode', $value);
            } elseif ($key === 'created_by' && $value) {
                $query->where('created_by', $value);
            } elseif ($key === 'created_from' && $value) {
                $query->whereDate('created_at', '>=', $value);
            } elseif ($key === 'created_to' && $value) {
                $query->whereDate('created_at', '<=', $value);
            } elseif ($key === 'order_by' && $value) {
                $orderDirection = $filters['order_direction'] ?? 'desc';
                $query->orderBy($value, $orderDirection);
            }
        }

        $perPage = $filters['per_page'] ?? 15;
        return $query->paginate($perPage);
    }

    public function findByUuid(string $uuid, array $with = [], array $withCount = []): ?LeadForm
    {
        $query = LeadForm::where('uuid', $uuid);

        if (!empty($with)) {
            $query->with($with);
        }

        if (!empty($withCount)) {
            $query->withCount($withCount);
        }

        return $query->first();
    }

    public function findByUuidOrFail(string $uuid, array $with = [], array $withCount = []): LeadForm
    {
        $query = LeadForm::where('uuid', $uuid);

        if (!empty($with)) {
            $query->with($with);
        }

        if (!empty($withCount)) {
            $query->withCount($withCount);
        }

        return $query->firstOrFail();
    }

    public function findActiveByUuid(string $uuid, array $with = []): ?LeadForm
    {
        $query = LeadForm::where('uuid', $uuid)->where('is_active', true);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->first();
    }

    public function findActiveByUuidOrFail(string $uuid, array $with = []): LeadForm
    {
        $query = LeadForm::where('uuid', $uuid)->where('is_active', true);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->firstOrFail();
    }

    public function create(array $data): LeadForm
    {
        return LeadForm::create($data);
    }

    public function update(LeadForm $leadForm, array $data): LeadForm
    {
        $leadForm->update($data);
        return $leadForm->fresh();
    }

    public function delete(LeadForm $leadForm): bool
    {
        return $leadForm->delete();
    }

    public function syncFields(LeadForm $leadForm, array $fieldData): void
    {
        $leadForm->formFields()->sync($fieldData);
    }

    public function syncMemberCategories(LeadForm $leadForm, array $categoryData): void
    {
        $leadForm->memberCategories()->sync($categoryData);
    }

    public function attachFields(LeadForm $leadForm, array $fieldData): void
    {
        $leadForm->formFields()->attach($fieldData);
    }

    public function attachMemberCategories(LeadForm $leadForm, array $categoryData): void
    {
        $leadForm->memberCategories()->attach($categoryData);
    }
}
