<?php

namespace App\Repositories\Eloquent\Lead;

use App\Models\Lead\Lead;
use App\Repositories\Contracts\Lead\LeadRepositoryInterface;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;

class LeadRepository implements LeadRepositoryInterface
{
    public function getAll(array $filters = [], array $with = []): LengthAwarePaginator
    {
        $query = Lead::query();

        // Apply relationships
        if (!empty($with)) {
            $query->with($with);
        }

        // Apply filters
        foreach ($filters as $key => $value) {
            if ($key === 'lead_form_id' && $value) {
                $query->where('lead_form_id', $value);
            } elseif ($key === 'lead_source' && $value) {
                $query->where('lead_source', $value);
            } elseif ($key === 'lead_statuses' && $value) {
                $query->whereIn('lead_status', $value);
            } elseif ($key === 'search' && $value) {
                $query->where(function ($q) use ($value) {
                    $q->where('first_name', 'like', '%' . $value . '%')
                      ->orWhere('last_name', 'like', '%' . $value . '%')
                      ->orWhere('email', 'like', '%' . $value . '%')
                      ->orWhere('phone', 'like', '%' . $value . '%');
                });
            } elseif ($key === 'email' && $value) {
                $query->where('email', $value);
            } elseif ($key === 'created_by' && $value) {
                $query->where('created_by', $value);
            } elseif ($key === 'assigned_to_user_id' && $value) {
                $query->whereHas('users', function ($q) use ($value) {
                    $q->where('user_id', $value);
                });
            } elseif ($key === 'created_from' && $value) {
                $query->whereDate('created_at', '>=', $value);
            } elseif ($key === 'created_to' && $value) {
                $query->whereDate('created_at', '<=', $value);
            } elseif ($key === 'with_trashed' && $value) {
                $query->withTrashed();
            } elseif ($key === 'only_trashed' && $value) {
                $query->onlyTrashed();
            } elseif ($key === 'order_by' && $value) {
                $orderDirection = $filters['order_direction'] ?? 'desc';
                $query->orderBy($value, $orderDirection);
            }
        }

        $perPage = $filters['per_page'] ?? 15;
        return $query->paginate($perPage);
    }

    public function findById(int $id): ?Lead
    {
        return Lead::find($id);
    }

    public function findByUuid(string $uuid, array $with = []): ?Lead
    {
        $query = Lead::where('uuid', $uuid);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->first();
    }

    public function findByUuidOrFail(string $uuid, array $with = []): Lead
    {
        $query = Lead::where('uuid', $uuid);

        if (!empty($with)) {
            $query->with($with);
        }

        return $query->firstOrFail();
    }

    public function create(array $data): Lead
    {
        return Lead::create($data);
    }

    public function update(Lead $lead, array $data): Lead
    {
        $lead->update($data);
        return $lead->fresh();
    }

    public function delete(Lead $lead): bool
    {
        return $lead->delete();
    }

    public function attachUsers(Lead $lead, array $pivotData): void
    {
        $lead->users()->attach($pivotData);
    }

    public function syncUsers(Lead $lead, array $pivotData): void
    {
        $lead->users()->sync($pivotData);
    }

    public function updateUserPivot(Lead $lead, int $userId, array $data): void
    {
        $lead->users()->updateExistingPivot($userId, $data);
    }

    public function setMeta(Lead $lead, string $key, $value): void
    {
        $lead->setMeta($key, $value);
    }

    public function setAllMeta(Lead $lead, array $meta): void
    {
        $lead->setMeta($meta);
    }
}
