<?php

namespace App\Repositories\Eloquent\Location;

use App\Models\Location\City;
use App\Repositories\Contracts\Location\CityRepositoryInterface;
use Illuminate\Database\Eloquent\Model;

class CityRepository implements CityRepositoryInterface
{
    public function getAll(array $args = [])
    {
        $query = City::query();
        if (!empty($args['search'])) {
            $query->where('city_name', 'like', '%' . $args['search'] . '%');
        }
        if (!empty($args['state_id'])) {
            $query->where('state_id', $args['state_id']);
        }
        if (!empty($args['country_id'])) {
            $query->where('country_id', $args['country_id']);
        }
        return $query;
    }

    public function find($id)
    {
        return City::find($id);
    }

    public function findBySlug(string $slug)
    {
        return City::where('slug', $slug)->first();
    }

    public function findByUuid(string $uuid)
    {
        return City::where('uuid', $uuid)->first();
    }

    public function create(array $data)
    {
        return City::create($data);
    }

    public function update($id, array $data)
    {
        $city = City::find($id);
        if ($city) {
            $city->update($data);
        }
        return $city;
    }

    public function delete($id)
    {
        return City::find($id)?->delete();
    }

    /**
     * Find a city by country ID and city name
     *
     * @param int $countryId
     * @param string $cityName
     * @return Model|null
     */
    public function findByCountryAndName(int $countryId, string $cityName): ?Model
    {
        return City::query()->whereHas('state', function ($query) use ($countryId) {
            $query->where('country_id', $countryId);
        })->where(function ($query) use ($cityName) {
            $query->where('city_name', $cityName)
                ->orWhere('city_name', 'LIKE', "%{$cityName}%");
        })->first();
    }

    public function findOrCreateByNameAndState(string $name, int $stateId, int $countryId)
    {
        // First try to find by name and state
        $city = City::where('city_name', $name)
            ->where('state_id', $stateId)
            ->first();
        if (!$city) {
            $city = City::create([
                'city_name' => $name,
                'state_id' => $stateId,
                'country_id' => $countryId
            ]);
        }

        return $city;
    }
}
