<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\Payment;

use App\Models\User\User;
use App\Models\Membership\Payment;
use App\Models\Membership\Purchase;
use App\Repositories\Contracts\Payment\PaymentRepositoryInterface;
use App\Enums\PaymentType;
use App\Enums\PaymentStatus;

class PaymentRepository implements PaymentRepositoryInterface
{
    /**
     * Create a payment record
     */
    public function createPayment(
        User $user,
        Purchase $purchase,
        string $paymentMethod,
        float $amount,
        string $currency,
        array $metadata = []
    ): Payment {
        return Payment::create([
            'user_id' => $user->id,
            'purchase_id' => $purchase->id,
            'payment_type' => PaymentType::Payment->value,
            'payment_method' => $paymentMethod,
            'amount' => $amount,
            'currency' => $currency,
            'status' => PaymentStatus::Pending->value,
            'payment_date' => now(),
            'service_response' => json_encode($metadata),
            'created_by' => $user->id,
        ]);
    }

    /**
     * Update payment status
     */
    public function updatePaymentStatus(Payment $payment, string $status, array $metadata = []): bool
    {
        return $payment->update([
            'status' => $status,
            'service_response' => json_encode($metadata),
        ]);
    }

    /**
     * Find payment by transaction ID
     */
    public function findByTransactionId(string $transactionId): ?Payment
    {
        return Payment::where('transaction_id', $transactionId)->first();
    }
}
