<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\Product;

use App\Models\Product\Product;
use App\Repositories\Contracts\Product\ProductRepositoryInterface;

class ProductRepository implements ProductRepositoryInterface
{
    public function getAll(array $args = [])
    {
        $query = Product::query();
        
        if (!empty($args['search'])) {
            $query->where(function($q) use ($args) {
                $q->where('product_title', 'like', '%' . $args['search'] . '%')
                  ->orWhere('product_description', 'like', '%' . $args['search'] . '%');
            });
        }

        if (!empty($args['category'])) {
            $query->whereHas('categories', function($q) use ($args) {
                $q->where('slug', $args['category']);
            });
        }

        if (!empty($args['created_by'])) {
            $query->where('created_by', $args['created_by']);
        }

        return $query;
    }

    public function find($id)
    {
        return Product::find($id);
    }

    public function findBySlug(string $slug)
    {
        return Product::where('slug', $slug)->first();
    }

    public function findByUuid(string $uuid)
    {
        return Product::where('uuid', $uuid)->first();
    }

    public function create(array $data)
    {
        return Product::create($data);
    }

    public function update($id, array $data)
    {
        $product = Product::find($id);
        if ($product) {
            $product->update($data);
        }
        return $product;
    }

    public function updateByUuid(string $uuid, array $data)
    {
        $product = Product::where('uuid', $uuid)->firstOrFail();
        $product->update($data);
        return $product;
    }

    public function delete($id)
    {
        return Product::find($id)?->delete();
    }

    public function getPublished()
    {
        return Product::published();
    }

    public function getChildProducts($parentId)
    {
        return Product::where('parent_id', $parentId)->get();
    }

    public function getProductsByCategory($categoryId)
    {
        return Product::whereHas('categories', function($query) use ($categoryId) {
            $query->where('product_categories.id', $categoryId);
        })->get();
    }
} 