<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\Project;

use App\Models\Project\Project;
use App\Repositories\Contracts\Project\ProjectRepositoryInterface;
use App\Models\Location\Country;
use App\Models\Location\City;
use App\Models\Location\State;

class ProjectRepository implements ProjectRepositoryInterface
{
    public function getAll($args = [])
    {
        $query = Project::query();

        if (!empty($args['search'])) {
            $query->where(function ($q) use ($args) {
                $q->where('title', 'like', '%' . $args['search'] . '%')
                    ->orWhere('description', 'like', '%' . $args['search'] . '%');
            });
        }

        if (!empty($args['type'])) {
            $query->whereHas('types', function ($q) use ($args) {
                $q->where('slug', $args['type']);
            });
        }

        if (!empty($args['status'])) {
            $query->where('project_status', $args['status']);
        }

        if (!empty($args['user_id'])) {
            $query->where('user_id', $args['user_id']);
        }

        return $query;
    }

    public function find($id)
    {
        return Project::find($id);
    }

    public function findBySlug(string $slug)
    {
        return Project::where('slug', $slug)->first();
    }

    public function findByUuid(string $uuid)
    {
        return Project::where('uuid', $uuid)->first();
    }

    public function create(array $data)
    {
        $project = Project::create($data);
        $this->saveLocation($project, $data);
        return $project;
    }

    public function update($id, array $data)
    {
        $project = Project::find($id);
        $project->update($data);
        $this->saveLocation($project, $data);
        return $project;
    }

    public function updateByUuid(string $uuid, array $data)
    {
        $project = Project::where('uuid', $uuid)->firstOrFail();
        $project->update($data);
        $this->saveLocation($project, $data);
        return $project;
    }

    public function delete($id)
    {
        return Project::find($id)->delete();
    }

    public function getRelatedProjects($args = [])
    {
        $project = $args['project'];
        $limit = $args['limit'] ?? 3;
        $types = $project->types->pluck('id');

        return Project::where('projects.id', '!=', $project->id)
            ->whereHas('types', function ($q) use ($types) {
                $q->whereIn('project_types.id', $types);
            })
            ->published()
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }

    public function getFeaturedProjects($args = [])
    {
        $limit = $args['limit'] ?? 3;

        return Project::when(!empty($args['project_status']), function ($q) use ($args) {
            $q->where('project_status', $args['project_status']);
        })
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }

    public function getRelatedProjectsByProjectId(int $projectId, int $limit = 3)
    {
        $project = Project::find($projectId);
        $types = $project->types->pluck('id');

        return Project::where('id', '!=', $projectId)
            ->where('user_id', $project->user_id)
            ->whereHas('types', function ($q) use ($types) {
                $q->whereIn('project_types.id', $types);
            })
            ->published()
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }

    public function saveLocation(Project $project, array $data)
    {
        $countryName = !empty($data['country_name']) ? $data['country_name'] : null;
        $countryCode = !empty($data['country_code']) ? $data['country_code'] : null;
        $cityName = !empty($data['city_name']) ? $data['city_name'] : null;
        $stateName = !empty($data['state_name']) ? $data['state_name'] : null;
        $stateCode = !empty($data['state_code']) ? $data['state_code'] : null;
        $location = !empty($data['location']) ? $data['location'] : null;
        $address = !empty($data['address']) ? $data['address'] : null;

        $project->location = $location;

        // Create country if not exists
        if ($countryName) {

            $country = Country::firstOrCreate(
                ['country_name' => $countryName],
                ['country_code' => $countryCode]
            );
            $project->country_id = $country->id;

            // Create state if not exists
            if ($stateName) {
                $state = State::firstOrCreate(
                    ['country_id' => $country->id, 'state_name' => $stateName],
                    ['state_code' => $stateCode]
                );
                $project->state_id = $state->id;
            }

            // Create city if not exists
            if ($cityName) {
                $city = City::firstOrCreate(
                    ['country_id' => $country->id, 'city_name' => $cityName],
                    ['state_id' => $state->id]
                );
                $project->city_id = $city->id;
            }
        }

        $project->save();
        return $project;
    }
}
