<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\Purchase;

use App\Models\User\User;
use App\Models\User\UserCreditHistory;
use App\Models\Membership\Purchase;
use App\Repositories\Contracts\Purchase\PurchaseRepositoryInterface;
use App\Enums\PurchaseItemType;
use Illuminate\Database\Eloquent\Model;

class PurchaseRepository implements PurchaseRepositoryInterface
{
    /**
     * Create a credit purchase record
     */
    public function createCreditPurchase(
        User $user,
        int $quantity,
        float $basePrice,
        float $amount,
        float $discountAmount,
        string $currency
    ): Purchase {
        return Purchase::create([
            'user_id' => $user->id,
            'item_type' => PurchaseItemType::Credit->value,
            'purchaseable_type' => UserCreditHistory::class,
            'purchaseable_id' => null, // Will be set after UserCreditHistory is created
            'amount' => $basePrice,
            'currency' => $currency,
            'quantity' => $quantity,
            'total_amount' => $amount,
            'discount_amount' => $discountAmount,
            'tax_amount' => 0.00,
            'purchase_date' => now(),
            'notes' => "Credit purchase: {$quantity} credits",
            'created_by' => $user->id,
        ]);
    }

    /**
     * Update purchase purchaseable_id
     */
    public function updatePurchaseableId(Purchase $purchase, int $purchaseableId): bool
    {
        return $purchase->update(['purchaseable_id' => $purchaseableId]);
    }

    /**
     * Get all purchases with optional filters
     */
    public function getAll(array $args = [])
    {
        $query = Purchase::query();

        // Eager load relationships if needed
        if (!empty($args['with_plan'])) {
            $query->with(['purchaseable.membershipPlan']);
        }

        // Filter by user id
        if (!empty($args['user_id'])) {
            $query->where('user_id', $args['user_id']);
        }
        // Filter by item type
        if (!empty($args['item_type'])) {
            $query->where('item_type', $args['item_type']);
        }

        // Filter by member plan (using slug)
        if (!empty($args['memberPlan'])) {
            $query->whereHas('purchaseable.membershipPlan', function ($subQuery) use ($args) {
                $subQuery->where('slug', $args['memberPlan']);
            });
        }

        // Filter by member plan ID (fallback)
        if (!empty($args['member_plan_id'])) {
            $query->where('member_plan_id', $args['member_plan_id']);
        }

        // Filter by membership plan features
        if (!empty($args['featured_plans_only'])) {
            $query->whereHas('purchaseable.membershipPlan', function ($subQuery) {
                $subQuery->where('is_featured', true);
            });
        }

        if (!empty($args['active_plans_only'])) {
            $query->whereHas('purchaseable.membershipPlan', function ($subQuery) {
                $subQuery->where('is_active', true);
            });
        }

        // Filter by date range using helper function
        if (!empty($args['dateRange'])) {
            if (is_numeric($args['dateRange'])) {
                // Use helper function for days filtering
                $dateRange = esp_get_date_range($args['dateRange']);
                $query->whereBetween('created_at', [
                    $dateRange['start_date']->startOfDay(),
                    $dateRange['end_date']->endOfDay()
                ]);
            } else if ($args['dateRange'] === 'custom') {
                // Handle custom date range with startDate and endDate
                if (!empty($args['startDate']) && !empty($args['endDate'])) {
                    $dateRange = esp_get_date_range($args['startDate'], $args['endDate']);
                    $query->whereBetween('created_at', [
                        $dateRange['start_date'] . ' 00:00:00',
                        $dateRange['end_date'] . ' 23:59:59'
                    ]);
                } else if (!empty($args['startDate'])) {
                    // Only start date provided
                    $query->whereDate('created_at', '>=', $args['startDate']);
                } else if (!empty($args['endDate'])) {
                    // Only end date provided
                    $query->whereDate('created_at', '<=', $args['endDate']);
                }
            }
        }

        return $query;
    }

    /**
     * Find purchase by ID
     */
    public function findById(int $id): ?Purchase
    {
        return Purchase::find($id);
    }

    /**
     * Find purchase by UUID
     */
    public function findByUuid(string $uuid): ?Purchase
    {
        return Purchase::where('uuid', $uuid)->first();
    }

    /**
     * Create a new purchase record
     */
    public function create(array $data): Purchase
    {
        return Purchase::create($data);
    }

    /**
     * Update an existing purchase record
     */
    public function update(Model $model, array $data): bool
    {
        return $model->update($data);
    }

    /**
     * Delete a purchase record
     */
    public function delete(Model $model): bool
    {
        return $model->delete();
    }

    /**
     * Delete purchase by ID
     */
    public function deleteById(int $id): bool
    {
        return Purchase::where('id', $id)->delete();
    }

    /**
     * Delete purchase by UUID
     */
    public function deleteByUuid(string $uuid): bool
    {
        return Purchase::where('uuid', $uuid)->delete();
    }
}
