<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\Review;

use App\Models\Review;
use App\Repositories\Contracts\Review\ReviewRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

class ReviewRepository implements ReviewRepositoryInterface
{
    public function getAll(array $args = []): Builder
    {
        $query = Review::query();

        if (!empty($args['search'])) {
            $query->where(function ($q) use ($args) {
                $q->where('review_title', 'like', '%' . $args['search'] . '%')
                    ->orWhere('review_content', 'like', '%' . $args['search'] . '%');
            });
        }

        if (isset($args['rating'])) {
            $query->where('review_rating', $args['rating']);
        }

        if (isset($args['status'])) {
            $query->where('status', $args['status']);
        }

        if (!empty($args['sort_by']) && !empty($args['sort_direction'])) {
            $query->orderBy($args['sort_by'], $args['sort_direction']);
        } else {
            $query->orderBy('created_at', 'desc');
        }

        return $query;
    }

    public function find(int $id): ?Model
    {
        return Review::find($id);
    }

    public function findByUuid(string $uuid): ?Model
    {
        return Review::where('uuid', $uuid)->first();
    }

    public function create(array $data): Model
    {
        return Review::create($data);
    }

    public function update(int $id, array $data): bool
    {
        $review = Review::find($id);
        if (!$review) {
            return false;
        }
        return $review->update($data);
    }

    public function updateByUuid(string $uuid, array $data): bool
    {
        $review = Review::where('uuid', $uuid)->first();
        if (!$review) {
            return false;
        }
        return $review->update($data);
    }

    public function delete(int $id): bool
    {
        $review = Review::find($id);
        if (!$review) {
            return false;
        }
        return $review->delete();
    }

    public function getReviewsByReviewable(string $type, int $id, array $args = []): Builder
    {
        $query = Review::query()
            ->where('reviewable_type', $type)
            ->where('reviewable_id', $id);

        if (isset($args['status'])) {
            $query->where('status', $args['status']);
        } else {
            $query->where('status', true);
        }

        if (!empty($args['sort_by']) && !empty($args['sort_direction'])) {
            $query->orderBy($args['sort_by'], $args['sort_direction']);
        } else {
            $query->orderBy('created_at', 'desc');
        }

        return $query;
    }
}
