<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\Team;

use App\Models\Team\TeamMember;
use App\Repositories\Contracts\Team\TeamMemberRepositoryInterface;

class TeamMemberRepository implements TeamMemberRepositoryInterface
{
    public function getAll($args = [])
    {
        $query = TeamMember::query();

        if (!empty($args['search'])) {
            $query->where(function ($q) use ($args) {
                $q->where('name', 'like', '%' . $args['search'] . '%')
                    ->orWhere('position', 'like', '%' . $args['search'] . '%')
                    ->orWhere('bio', 'like', '%' . $args['search'] . '%');
            });
        }

        if (!empty($args['role'])) {
            $query->where('role', $args['role']);
        }

        if (!empty($args['project'])) {
            $query->whereHas('projects', function ($q) use ($args) {
                $q->where('projects.id', $args['project']);
            });
        }

        if (!empty($args['order_by'])) {
            $query->orderBy($args['order_by'], $args['order_direction'] ?? 'asc');
        }

        if (!empty($args['user_id'])) {
            $query->where('user_id', $args['user_id']);
        }

        return $query;
    }

    public function find($id)
    {
        return TeamMember::find($id);
    }

    public function findBySlug(string $slug)
    {
        return TeamMember::where('slug', $slug)->first();
    }

    public function findByUuid(string $uuid)
    {
        return TeamMember::where('uuid', $uuid)->first();
    }

    public function create(array $data)
    {
        return TeamMember::create($data);
    }

    public function update($id, array $data)
    {
        $member = TeamMember::find($id);
        $member->update($data);
        return $member;
    }

    public function updateByUuid(string $uuid, array $data)
    {
        $member = TeamMember::where('uuid', $uuid)->firstOrFail();
        $member->update($data);
        return $member;
    }

    public function delete($id)
    {
        return TeamMember::find($id)->delete();
    }

    public function getTeamMembers(array $args = [])
    {
        $query = TeamMember::query();

        if (!empty($args['project_id'])) {
            $query->whereHas('projects', function ($q) use ($args) {
                $q->where('projects.id', $args['project_id']);
            });
        }

        if (!empty($args['role'])) {
            $query->where('role', $args['role']);
        }

        $query->when(!empty($args['order']), function ($q) use ($args) {
            $q->orderBy('order', $args['order']);
        });

        if (!empty($args['limit'])) {
            $query->limit($args['limit']);
        }

        return $query->get();
    }

    public function getFeaturedMembers(array $args = [])
    {
        $limit = $args['limit'] ?? 3;

        return TeamMember::where('is_featured', true)
            ->when(!empty($args['role']), function ($q) use ($args) {
                $q->where('role', $args['role']);
            })
            ->orderBy('order')
            ->limit($limit)
            ->get();
    }
}
