<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\User;

use App\Models\User\Inquiry;
use App\Repositories\Contracts\User\InquiryRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

class InquiryRepository implements InquiryRepositoryInterface
{
    public function getAll(array $args = []): Builder
    {
        $query = Inquiry::query()->with(['contact', 'user', 'creator', 'inquiryable']);

        if (!empty($args['search'])) {
            $query->where(function ($q) use ($args) {
                $q->whereHas('contact', function ($contactQuery) use ($args) {
                    $contactQuery->where('first_name', 'like', '%' . $args['search'] . '%')
                        ->orWhere('last_name', 'like', '%' . $args['search'] . '%')
                        ->orWhere('email', 'like', '%' . $args['search'] . '%');
                })
                    ->orWhere('form_data', 'like', '%' . $args['search'] . '%');
            });
        }

        if (!empty($args['status'])) {
            $query->where('inquiry_status', $args['status']);
        }

        if (!empty($args['type'])) {
            $query->where('inquiry_type', $args['type']);
        }

        if (!empty($args['inquiryable_type'])) {
            $query->where('inquiryable_type', $args['inquiryable_type']);
        }

        if (!empty($args['contact_id'])) {
            $query->where('contact_id', $args['contact_id']);
        }

        return $query->orderBy('created_at', 'desc');
    }

    public function find(int $id): ?Model
    {
        return Inquiry::with(['contact', 'user', 'creator', 'inquiryable'])->find($id);
    }

    public function findByUuid(string $uuid): ?Model
    {
        return Inquiry::with(['contact', 'user', 'creator', 'inquiryable'])
            ->where('uuid', $uuid)->first();
    }

    public function create(array $data): Model
    {
        return Inquiry::create($data);
    }

    public function update(int $id, array $data): bool
    {
        $inquiry = Inquiry::find($id);
        if (!$inquiry) {
            return false;
        }
        return $inquiry->update($data);
    }

    public function updateByUuid(string $uuid, array $data): bool
    {
        $inquiry = Inquiry::where('uuid', $uuid)->first();
        if (!$inquiry) {
            return false;
        }
        return $inquiry->update($data);
    }

    public function delete(int $id): bool
    {
        $inquiry = Inquiry::find($id);
        if (!$inquiry) {
            return false;
        }
        return $inquiry->delete();
    }

    public function getByStatus(string $status): Builder
    {
        return Inquiry::with(['contact', 'user', 'creator', 'inquiryable'])
            ->where('inquiry_status', $status);
    }

    public function getByType(string $type): Builder
    {
        return Inquiry::with(['contact', 'user', 'creator', 'inquiryable'])
            ->where('inquiry_type', $type);
    }

    public function getByInquiryable(string $type, int $id): Builder
    {
        return Inquiry::with(['contact', 'user', 'creator'])
            ->where('inquiryable_type', $type)
            ->where('inquiryable_id', $id);
    }

    public function getByContact(int $contactId): Builder
    {
        return Inquiry::with(['user', 'creator', 'inquiryable'])
            ->where('contact_id', $contactId);
    }
}
