<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\User;

use App\Models\User\UserFavorite;
use App\Repositories\Contracts\User\UserFavoriteRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;

class UserFavoriteRepository implements UserFavoriteRepositoryInterface
{
    public function getAll(array $args = []): Builder
    {
        $query = UserFavorite::query();

        if (!empty($args['user_id'])) {
            $query->where('user_id', $args['user_id']);
        }

        if (!empty($args['model_type'])) {
            $query->where('model_type', $args['model_type']);
        }

        return $query->with('model');
    }

    public function find(int $id): ?Model
    {
        return UserFavorite::find($id);
    }

    public function create(array $data): Model
    {
        return UserFavorite::create($data);
    }

    public function delete(int $id): bool
    {
        return UserFavorite::destroy($id) > 0;
    }

    public function getUserFavorites(int $userId, string $modelType = null): Builder
    {
        $query = UserFavorite::where('user_id', $userId)
            ->with('model');

        if ($modelType) {
            $query->where('model_type', $modelType);
        }

        return $query->orderBy('created_at', 'desc');
    }

    public function getUserFavoritesPaginated(int $userId, string $modelType = null, int $perPage = 15): LengthAwarePaginator
    {
        return $this->getUserFavorites($userId, $modelType)->paginate($perPage);
    }

    public function removeFavorite(int $userId, string $modelType, int $modelId): bool
    {
        return UserFavorite::where('user_id', $userId)
            ->where('model_type', $modelType)
            ->where('model_id', $modelId)
            ->delete() > 0;
    }

    public function isFavorite(int $userId, string $modelType, int $modelId): bool
    {
        return UserFavorite::where('user_id', $userId)
            ->where('model_type', $modelType)
            ->where('model_id', $modelId)
            ->exists();
    }
}