<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\User;

use App\Enums\MembershipType;
use App\Enums\SubscriptionStatus;
use App\Models\User\User;
use App\Repositories\Contracts\User\UserRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class UserRepository implements UserRepositoryInterface
{
    public function getAll(array $args = []): Builder
    {
        $query = User::query();
        if (! empty($args['search'])) {
            $query->where(function ($q) use ($args) {
                $q->where('users.name', 'like', '%' . $args['search'] . '%')
                    ->orWhere('users.email', 'like', '%' . $args['search'] . '%')
                    ->orWhereHas('profile', function ($profileQuery) use ($args) {
                        $profileQuery->where('company_name', 'like', '%' . $args['search'] . '%');
                    });
            });
        }

        if (! empty($args['role'])) {
            $roles = is_array($args['role']) ? $args['role'] : [$args['role']];
            $query->whereHas('roles', function ($q) use ($roles) {
                // dump($roles);
                $q->whereIn('name', $roles);
            });
        }

        if (! empty($args['categories'])) {
            $query->whereHas('memberCategories', function ($q) use ($args) {
                $categories = (array) $args['categories'];
                $q->whereIn('member_categories.id', $categories);
            });
        }

        // Location based filtering
        if (! empty($args['radius']) && ! empty($args['lat']) && ! empty($args['lng'])) {
            // Use radius search with the provided coordinates
            $radius = (float) $args['radius'];
            $lat    = (float) $args['lat'];
            $lng    = (float) $args['lng'];

            $query->whereHas('primaryLocation', function ($q) use ($lat, $lng, $radius) {
                // Haversine formula for distance calculation
                $q->selectRaw('
                    (6371 * acos(
                        cos(radians(?)) *
                        cos(radians(latitude)) *
                        cos(radians(longitude) - radians(?)) +
                        sin(radians(?)) *
                        sin(radians(latitude))
                    )) AS distance', [$lat, $lng, $lat])
                    ->havingRaw('distance <= ?', [$radius]);
            });

            // Add the distance calculation to the main query for sorting
            $query->select('users.*')
                ->addSelect(DB::raw('(
                    6371 * acos(
                        cos(radians(' . $lat . ')) *
                        cos(radians(user_locations.latitude)) *
                        cos(radians(user_locations.longitude) - radians(' . $lng . ')) +
                        sin(radians(' . $lat . ')) *
                        sin(radians(user_locations.latitude))
                    )
                ) AS distance'))
                ->join('user_locations', 'users.id', '=', 'user_locations.user_id')
                ->where('user_locations.primary_location', true)
                ->orderBy('distance', 'asc');
        } else {
            if (! empty($args['city'])) {
                $query->whereHas('primaryLocation', function ($q) use ($args) {
                    $q->where('city_id', $args['city']);
                });
            }
            if (! empty($args['state'])) {
                $query->whereHas('primaryLocation', function ($q) use ($args) {
                    $q->where('state_id', $args['state']);
                });
            }

            if (! empty($args['country'])) {
                $query->whereHas('primaryLocation', function ($q) use ($args) {
                    $q->where('country_id', $args['country']);
                });
            }
        }

        if (!empty($args['membershipType'])) {
            $query->whereHas('activeMembershipPlans', function ($q) use ($args) {
                $q->where('membership_type', $args['membershipType']);
            });
        }

        if (! empty($args['rating'])) {
            // Handle array of ratings for range filtering
            if (is_array($args['rating'])) {
                $query->whereHas('profile', function ($q) use ($args) {
                    $q->where(function ($subQuery) use ($args) {
                        foreach ($args['rating'] as $rating) {
                            $rating = (int) $rating;
                            if ($rating == 1) {
                                $subQuery->orWhereBetween('profile_rating', [0, 1.9]);
                            } elseif ($rating == 2) {
                                $subQuery->orWhereBetween('profile_rating', [2.0, 2.9]);
                            } elseif ($rating == 3) {
                                $subQuery->orWhereBetween('profile_rating', [3.0, 3.9]);
                            } elseif ($rating == 4) {
                                $subQuery->orWhereBetween('profile_rating', [4.0, 4.9]);
                            } elseif ($rating == 5) {
                                $subQuery->orWhere('profile_rating', 5);
                            }
                        }
                    });
                });
            } else {
                // Handle single rating (backward compatibility)
                $query->withAvg('reviews', 'review_rating')
                    ->having('reviews_avg_review_rating', '>=', $args['rating']);
            }
        }

        if (! empty($args['features'])) {
            $features = (array) $args['features'];

            if (in_array('emergency', $features)) {
                $query->whereHas('profile', function ($q) {
                    $q->whereExists(function ($query) {
                        $query->from('user_profiles_meta')
                            ->whereColumn('user_profiles_meta.user_profile_id', 'user_profiles.id')
                            ->where('user_profiles_meta.key', 'emergency_service_enabled')
                            ->where('user_profiles_meta.value', '1');
                    });
                });
            }

            if (in_array('reviews', $features)) {
                $query->whereHas('reviews', function ($q) {
                    $q->where('reviews.review_rating', '>', 0);
                });
            }

            if (in_array('open_now', $features)) {
                $now         = \Carbon\Carbon::now();
                $currentDay  = strtolower($now->format('l'));
                $currentTime = $now->format('H:i');

                $query->whereHas('businessHours', function ($q) use ($currentDay, $currentTime) {
                    $q->where('day_of_week', $currentDay)
                        ->where('is_open', true)
                        ->where('open_time', '<=', $currentTime)
                        ->where('close_time', '>=', $currentTime);
                });
            }

            // Note: We don't filter for 'reviews' as per requirement
        }

        if (! empty($args['order_by'])) {
            if ($args['order_by'] == 'profile_rating') {
                $query->orderBy(
                    \App\Models\User\UserProfile::select('profile_rating')
                        ->whereColumn('user_profiles.user_id', 'users.id')
                        ->limit(1),
                    $args['order']
                );
            } else if ($args['order_by'] == 'featured_first') {
                $query->leftJoin('subscriptions', function ($join) {
                    $join->on('users.id', '=', 'subscriptions.user_id')
                        ->where('subscriptions.status', '=', SubscriptionStatus::Active->value);
                })
                    ->leftJoin('membership_plans', 'subscriptions.membership_plan_id', '=', 'membership_plans.id')
                    ->select('users.*')
                    ->orderByRaw("CASE WHEN membership_plans.membership_type = ? THEN 1 ELSE 0 END DESC", [MembershipType::Featured->value])
                    ->orderBy('users.created_at', 'desc');
            } else {
                $query->orderBy($args['order_by'], $args['order']);
            }
        } else {
            $query->orderBy('created_at', 'desc');
        }
        return $query;
    }

    public function find(int $id): ?Model
    {
        return User::find($id);
    }

    public function findBySlug(string $slug): ?Model
    {
        return User::where('slug', $slug)->first();
    }

    public function findByEmail(string $email): ?Model
    {
        return User::where('email', $email)->first();
    }

    public function create(array $data): Model
    {
        return User::create($data);
    }

    public function update(int $id, array $data): bool
    {
        return User::find($id)->update($data);
    }

    public function delete(int $id): bool
    {
        return User::find($id)->delete();
    }
}
