<?php

declare(strict_types=1);

namespace App\Repositories\Eloquent\User;

use App\Models\User\UserService;
use App\Repositories\Contracts\User\UserServiceRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

class UserServiceRepository implements UserServiceRepositoryInterface
{
    public function getAll(array $args = []): Builder
    {
        $query = UserService::query();

        if (!empty($args['search'])) {
            $query->where(function ($q) use ($args) {
                $q->where('service_title', 'like', '%' . $args['search'] . '%')
                    ->orWhere('service_description', 'like', '%' . $args['search'] . '%');
            });
        }

        return $query;
    }

    public function find(int $id): ?Model
    {
        return UserService::find($id);
    }

    public function findByUuid(string $uuid): ?Model
    {
        return UserService::where('uuid', $uuid)->first();
    }

    public function create(array $data): Model
    {
        // Set initial order if not provided
        if (!isset($data['service_order'])) {
            $maxOrder = UserService::where('user_id', $data['user_id'])->max('service_order') ?? 0;
            $data['service_order'] = $maxOrder + 1;
        }

        return UserService::create($data);
    }

    public function update(int $id, array $data): bool
    {
        return UserService::find($id)->update($data);
    }

    public function updateByUuid(string $uuid, array $data): bool
    {
        $service = UserService::where('uuid', $uuid)->first();
        if (!$service) {
            return false;
        }
        return $service->update($data);
    }

    public function delete(int $id): bool
    {
        return UserService::find($id)->delete();
    }

    public function getUserServices(int $userId): Builder
    {
        return UserService::where('user_id', $userId)
            ->orderBy('service_order', 'asc')
            ->orderBy('created_at', 'desc');
    }
}
