<?php

declare(strict_types=1);

namespace App\Repositories\Location;

use App\Models\Location\Area;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;

class AreaRepository
{
    /**
     * Find or create area by name and city ID
     */
    public function findOrCreateByNameAndCity(string $name, int $cityId): Area
    {
        $slug = Str::slug($name);
        
        return Area::firstOrCreate(
            [
                'name' => $name,
                'city_id' => $cityId,
            ],
            [
                'slug' => $slug,
            ]
        );
    }

    /**
     * Get areas by city ID
     */
    public function getByCity(int $cityId): Collection
    {
        return Area::where('city_id', $cityId)
            ->orderBy('name')
            ->get();
    }

    /**
     * Search areas by name
     */
    public function search(string $search, ?int $cityId = null): Collection
    {
        $query = Area::where('name', 'LIKE', "%{$search}%");
        
        if ($cityId) {
            $query->where('city_id', $cityId);
        }
        
        return $query->orderBy('name')
            ->limit(20)
            ->get();
    }

    /**
     * Find area by ID
     */
    public function findById(int $id): ?Area
    {
        return Area::find($id);
    }

    /**
     * Find area by UUID
     */
    public function findByUuid(string $uuid): ?Area
    {
        return Area::where('uuid', $uuid)->first();
    }

    /**
     * Get all areas with city information
     */
    public function getAllWithCity(): Collection
    {
        return Area::with('city')->orderBy('name')->get();
    }
}