<?php

namespace App\Services;

use App\Models\User\UserFavorite;
use Illuminate\Support\Facades\Auth;


class ActionHandlerService
{
    protected $modelMap = [
        'classified' => \App\Models\Classified\Classified::class,
        'job' => \App\Models\Job\Job::class,
        'project' => \App\Models\Project\Project::class,
        'equipment' => \App\Models\Equipment\Equipment::class,
        'article' => \App\Models\Blog\Post::class,
        'member' => \App\Models\User\User::class,
    ];

    public function handleAction(string $action, string $module, string $modelId, array $additionalData = []): array
    {
        // Verify module exists
        if (!isset($this->modelMap[$module])) {
            throw new \InvalidArgumentException("Invalid module: {$module}");
        }

        $modelClass = $this->modelMap[$module];
        $model = $modelClass::where('uuid', $modelId)->first();

        // Route to appropriate action handler
        switch ($action) {
            case 'add_to_favourite':
                return $this->handleAddToFavourite($model, $module, $additionalData);

            case 'remove_from_favourite':
                return $this->handleRemoveFromFavourite($model, $module, $additionalData);
            default:
                throw new \InvalidArgumentException("Invalid action: {$action}");
        }
    }

    protected function handleAddToFavourite($model, string $module, array $data): array
    {
        $userId = Auth::id();

        if (!$userId) {
            throw new \Exception('User must be authenticated');
        }

        // Check if already favorited
        $existing = UserFavorite::where([
            'user_id' => $userId,
            'model_type' => get_class($model),
            'model_id' => $model->id,
        ])->first();

        if ($existing) {
            return [
                'message' => 'Already added to favorites',
                'data' => ['is_favorite' => true]
            ];
        }

        UserFavorite::create([
            'user_id' => $userId,
            'model_type' => get_class($model),
            'model_id' => $model->id,
        ]);

        return [
            'message' => 'Added to favorites successfully',
            'data' => ['is_favorite' => true]
        ];
    }

    protected function handleRemoveFromFavourite($model, string $module, array $data): array
    {
        $userId = Auth::id();

        if (!$userId) {
            throw new \Exception('User must be authenticated');
        }

        $deleted = UserFavorite::where([
            'user_id' => $userId,
            'model_type' => get_class($model),
            'model_id' => $model->id,
        ])->delete();

        return [
            'message' => $deleted ? 'Removed from favorites' : 'Not in favorites',
            'data' => ['is_favorite' => false]
        ];
    }
}
