<?php

namespace App\Services\ApiClient;

use App\DTOs\ApiClient\CreateApiClientDTO;
use App\DTOs\ApiClient\UpdateApiClientDTO;
use App\DTOs\ApiClient\GenerateTokenDTO;
use App\Models\ApiClient;
use App\Repositories\Contracts\ApiClient\ApiClientRepositoryInterface;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Laravel\Sanctum\NewAccessToken;

class ApiClientService
{
    public function __construct(
        protected ApiClientRepositoryInterface $apiClientRepository
    ) {}

    /**
     * Get all API clients with optional filters
     */
    public function getAll(array $filters = [], array $with = []): LengthAwarePaginator
    {
        return $this->apiClientRepository->getAll($filters, $with);
    }

    /**
     * Get all active API clients
     */
    public function getAllActive(array $with = []): Collection
    {
        return $this->apiClientRepository->getAllActive($with);
    }

    /**
     * Find API client by ID
     */
    public function findById(int $id, array $with = []): ?ApiClient
    {
        return $this->apiClientRepository->findById($id, $with);
    }

    /**
     * Find API client by ID or fail
     */
    public function findByIdOrFail(int $id, array $with = []): ApiClient
    {
        return $this->apiClientRepository->findByIdOrFail($id, $with);
    }

    /**
     * Create a new API client
     */
    public function create(CreateApiClientDTO $dto): ApiClient
    {
        return $this->apiClientRepository->create($dto->toArray());
    }

    /**
     * Update an API client
     */
    public function update(int $id, UpdateApiClientDTO $dto): ApiClient
    {
        $apiClient = $this->findByIdOrFail($id);

        if (!$dto->hasChanges()) {
            return $apiClient;
        }

        return $this->apiClientRepository->update($apiClient, $dto->toArray());
    }

    /**
     * Delete an API client
     */
    public function delete(int $id): bool
    {
        $apiClient = $this->findByIdOrFail($id);

        // Revoke all tokens before deleting
        $this->revokeAllTokens($id);

        return $this->apiClientRepository->delete($apiClient);
    }

    /**
     * Activate an API client
     */
    public function activate(int $id): ApiClient
    {
        $apiClient = $this->findByIdOrFail($id);
        return $this->apiClientRepository->activate($apiClient);
    }

    /**
     * Deactivate an API client
     */
    public function deactivate(int $id): ApiClient
    {
        $apiClient = $this->findByIdOrFail($id);
        return $this->apiClientRepository->deactivate($apiClient);
    }

    /**
     * Generate a new API token for the client
     */
    public function generateToken(int $id, GenerateTokenDTO $dto): NewAccessToken
    {
        $apiClient = $this->findByIdOrFail($id);

        if (!$apiClient->is_active) {
            throw new \Exception('Cannot generate token for inactive API client');
        }

        return $apiClient->createToken(
            $dto->getTokenName(),
            $dto->getAbilities()
        );
    }

    /**
     * Revoke all tokens for an API client
     */
    public function revokeAllTokens(int $id): void
    {
        $apiClient = $this->findByIdOrFail($id);
        $this->apiClientRepository->revokeAllTokens($apiClient);
    }

    /**
     * Check if an API client can perform a specific action
     */
    public function canPerformAction(ApiClient $apiClient, string $ability): bool
    {
        return $apiClient->tokenCan($ability);
    }

    /**
     * Get token statistics for an API client
     */
    public function getTokenStatistics(int $id): array
    {
        $apiClient = $this->findByIdOrFail($id);

        return [
            'total_tokens' => $apiClient->tokens()->count(),
            'tokens' => $apiClient->tokens()->get()->map(function ($token) {
                return [
                    'id' => $token->id,
                    'name' => $token->name,
                    'abilities' => $token->abilities,
                    'last_used_at' => $token->last_used_at?->toIso8601String(),
                    'created_at' => $token->created_at->toIso8601String(),
                ];
            }),
        ];
    }
}
