<?php

namespace App\Services\Blog;

use App\Repositories\Contracts\Blog\PostRepositoryInterface;
use Illuminate\Pagination\LengthAwarePaginator;

class PostService
{
    protected $postRepository;

    public function __construct(PostRepositoryInterface $postRepository)
    {
        $this->postRepository = $postRepository;
    }

    public function getAll($args = [])
    {
        return $this->postRepository->getAll($args);
    }

    public function find($id)
    {
        return $this->postRepository->find($id);
    }

    public function findBySlug($slug)
    {
        return $this->postRepository->findBySlug($slug);
    }

    public function findByUuid(string $uuid)
    {
        return $this->postRepository->findByUuid($uuid);
    }

    public function create(array $data)
    {
        return $this->postRepository->create($data);
    }

    public function update($id, array $data)
    {
        return $this->postRepository->update($id, $data);
    }

    public function updateByUuid(string $uuid, array $data)
    {
        return $this->postRepository->updateByUuid($uuid, $data);
    }

    public function getRelatedPosts($args = [])
    {
        return $this->postRepository->getRelatedPosts($args);
    }

    public function getFeaturedPosts($args = [])
    {
        return $this->postRepository->getFeaturedPosts($args);
    }

    public function getPostsList(array $args = [])
    {
        // dump($args);
        // DB::enableQueryLog();
        $postCollection = $this->getAll($args)->paginate($args['limit'] ?? 10);
        // Log::info(DB::getQueryLog());
        // DB::disableQueryLog();
        return $this->sortUsersByPostsPriority($postCollection);
    }

    /**
     * Sort users by membership plan priority on paginated collection
     * Elite = 1 (highest priority)
     * Pro = 2 (medium priority) 
     * Free = 3 (lowest priority)
     *
     * @param LengthAwarePaginator $users The paginated users collection
     * @return LengthAwarePaginator The paginated collection with membership sorting applied
     */
    public function sortUsersByPostsPriority(LengthAwarePaginator $posts): LengthAwarePaginator
    {
        // Sort the collection
        $sortedPosts = $posts->getCollection()->sortBy(function ($post) {

            // Posts with is_featured = 1 come first
            if (!empty($post->is_featured) && $post->is_featured == 1) {
                return 1;
            }

            // Posts with published_at less than 24 hours ago come second
            $publicationTime = \Carbon\Carbon::parse($post->published_at);
            if ($publicationTime->diffInHours() < 24) {
                return 2;
            }

            // Then sort by published_at priority
            if (!empty($post->published_at) && $post->published_at) {
                return 3;
            }

            // Users without published_at come last


            // Users without membership plan come last
            return 0;
        });

        // Re-create the paginator with the sorted collection
        return new LengthAwarePaginator(
            $sortedPosts,
            $posts->total(),
            $posts->perPage(),
            $posts->currentPage(),
            ['path' => \Illuminate\Pagination\Paginator::resolveCurrentPath()]
        );
    }
}
