<?php

declare(strict_types=1);

namespace App\Services\Classified;

use App\Repositories\Contracts\Classified\ClassifiedRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use App\Repositories\Contracts\Classified\ClassifiedTypeRepositoryInterface;
use App\Repositories\Contracts\Location\StateRepositoryInterface;
use App\Repositories\Contracts\Location\CountryRepositoryInterface;
use App\Repositories\Contracts\Location\CityRepositoryInterface;
use App\Repositories\Contracts\Classified\ClassifiedCategoryRepositoryInterface;
use App\Repositories\Contracts\Classified\ClassifiedDeliveryRepositoryInterface;

class ClassifiedService
{
    public function __construct(
        protected ClassifiedRepositoryInterface $classifiedRepository,
        protected ClassifiedTypeRepositoryInterface $classifiedTypeRepository,
        protected StateRepositoryInterface $stateRepository,
        protected CountryRepositoryInterface $countryRepository,
        protected CityRepositoryInterface $cityRepository,
        protected ClassifiedCategoryRepositoryInterface $classifiedCategoryRepository,
        protected ClassifiedDeliveryRepositoryInterface $classifiedDeliveryRepository
    ) {}

    public function getAll(array $args = []): Builder
    {
        // Convert state slug to ID
        if (!empty($args['state']) && !is_numeric($args['state'])) {
            $state = $this->stateRepository->findBySlug($args['state']);
            if ($state) {
                $args['state'] = $state->id;
            }
        }

        // Convert country slug to ID
        if (!empty($args['country']) && !is_numeric($args['country'])) {
            $country = $this->countryRepository->findBySlug($args['country']);
            if ($country) {
                $args['country'] = $country->id;
            }
        }

        // Convert city slug to ID
        if (!empty($args['city']) && !is_numeric($args['city'])) {
            if (is_numeric($args['country'])) {
                $city = $this->cityRepository->findByCountryAndName($args['country'], $args['city']);
                if ($city) {
                    $args['city'] = $city->id;
                } else {
                    $args['city'] = null;
                }
            } else {
                $args['city'] = null;
            }
        }

        // Convert type slugs to IDs
        if (!empty($args['type'])) {
            $typeIds = [];
            foreach ($args['type'] as $typeSlug) {
                if (!is_numeric($typeSlug)) {
                    $type = $this->classifiedTypeRepository->findBySlug($typeSlug);
                    if ($type) {
                        $typeIds[] = $type->id;
                    }
                } else {
                    $typeIds[] = (int) $typeSlug;
                }
            }
            $args['type'] = $typeIds;
        }

        // Convert main category slug to ID
        if (!empty($args['categories'])) {
            $categoryIds = [];
            $categories = (array)$args['categories'];
            
            foreach ($categories as $category) {
                if (!is_numeric($category)) {
                    $categoryModel = $this->classifiedCategoryRepository->findBySlug((string)$category);
                    if ($categoryModel) {
                        $categoryIds[] = $categoryModel->id;
                    }
                } else {
                    $categoryIds[] = (int)$category;
                }
            }
            
            $args['categories'] = $categoryIds;
        }

        if (!empty($args['deliveries'])) {
            $args['deliveries'] = (array) $args['deliveries'];
            $deliveryIds = [];
            foreach ($args['deliveries'] as $deliverySlug) {
                if (!is_numeric($deliverySlug)) {
                    $delivery = $this->classifiedDeliveryRepository->findBySlug($deliverySlug);
                    if ($delivery) {
                        $deliveryIds[] = $delivery->id;
                    }
                } else {
                    $deliveryIds[] = (int) $deliverySlug;
                }
            }
            $args['deliveries'] = $deliveryIds;
        }


        return $this->classifiedRepository->getAll($args);
    }

    public function getScheduledForPublishing(): Builder
    {
        return $this->classifiedRepository->getScheduledForPublishing();
    }

    public function find(int $id): ?Model
    {
        return $this->classifiedRepository->find($id);
    }

    public function findBySlug(string $slug): ?Model
    {
        return $this->classifiedRepository->findBySlug($slug);
    }

    public function create(array $data): Model
    {
        return $this->classifiedRepository->create($data);
    }

    public function update(int $id, array $data): bool
    {
        return $this->classifiedRepository->update($id, $data);
    }

    public function delete(int $id): bool
    {
        return $this->classifiedRepository->delete($id);
    }
    public function findByUuid(string $uuid)
    {
        return $this->classifiedRepository->findByUuid($uuid);
    }
    public function updateByUuid(string $uuid, array $data)
    {
        return $this->classifiedRepository->updateByUuid($uuid, $data);
    }

    public function getClassifiedList(array $args = [])
    {
        $args = $this->filterSearchParams($args);
        $paginatedClassifieds = $this->getAll($args)
            ->has('user')
            ->orderBy('created_at', 'desc')
            ->paginate($args['limit'] ?? 15);

        return $this->sortFeaturedFirst($paginatedClassifieds);
    }

    /**
     * Filter search params
     * 
     * @param array $args
     * @return array
     */
    public function filterSearchParams(array $args = []): array
    {
        // Convert sub category slug to ID
        if (!empty($args['subCategory'])) {
            // If sub category is provided, set it as the main category
            $args['categories'] = $args['subCategory'];
        } elseif (!empty($args['mainCategory'])) {
            // If main category is provided, set it as the main category
            $args['categories'] = $args['mainCategory'];
        }
        if (!empty($args['delivery'])) {
            $args['deliveries'] = (array) $args['delivery'];
        }

        // Convert Item age from years to days
        if (!empty($args['itemAge'])) {
            $itemAgeRange = ['range' => $args['itemAge'], 'from' => 0, 'to' => 0];

            if ($args['itemAge'] == 'new' || $args['itemAge'] == '100+') {
                if ($args['itemAge'] == 'new') {
                    $itemAgeRange['from'] = 0;
                    $itemAgeRange['to'] = 30;
                } else {
                    $itemAgeRange['from'] = 36500;
                    $itemAgeRange['to'] = null;
                }
            } else {
                $itemAgeRange = explode('-', $args['itemAge']);

                $itemRangeFrom = isset($itemAgeRange[0]) ? (int) ($itemAgeRange[0]) * 365 : null;
                $itemRangeTo = isset($itemAgeRange[1]) ? (int) ($itemAgeRange[1]) * 365 : null;
                $itemAgeRange['from'] = $itemRangeFrom;
                $itemAgeRange['to'] = $itemRangeTo;
            }
            $args['itemAge'] = $itemAgeRange;
        }
        return $args;
    }

    /**
     * Sort paginated collection to show featured classifieds first
     *
     * @param mixed $paginator The paginated collection
     * @return mixed The paginator with sorted items
     */
    protected function sortFeaturedFirst($paginator)
    {
        // Sort the collection items to show featured classifieds first, then by created_at desc
        $sortedItems = $paginator->getCollection()->sortByDesc(function ($classified) {
            // Featured classifieds get priority (1), then sort by created_at timestamp
            return [
                $classified->is_featured ? 1 : 0,
                $classified->created_at->timestamp
            ];
        })->values();

        // Replace the collection in the paginator with the sorted one
        $paginator->setCollection($sortedItems);

        return $paginator;
    }
}
