<?php

declare(strict_types=1);

namespace App\Services\Credit;

use App\Models\User\User;
use App\Models\User\UserCreditHistory;
use App\Models\Membership\Purchase;
use App\Models\Membership\Payment;
use App\Services\Payment\StripePaymentService;
use App\Services\Payment\PayPalPaymentService;
use App\Services\Purchase\PurchaseService;
use App\Services\Payment\PaymentService;
use App\Enums\PaymentStatus;
use App\Enums\TransactionType;
use App\Enums\PurchaseItemType;
use App\Enums\PaymentMethod;
use App\Enums\PaymentType;
use Illuminate\Support\Facades\DB;
use Exception;

class CreditPurchaseService
{
    public function __construct(
        private StripePaymentService $stripePaymentService,
        private PayPalPaymentService $payPalPaymentService,
        private PurchaseService $purchaseService,
        private PaymentService $paymentService
    ) {}

    /**
     * Calculate the total price for credits including bulk discounts
     */
    public function calculateCreditPrice(int $quantity): array
    {
        $basePrice = (float) get_credit_price();
        $currency = get_credit_currency();
        $bulkDiscountsEnabled = is_bulk_discounts_enabled();

        $subtotal = $quantity * $basePrice;
        $discount = 0;
        $discountPercentage = 0;

        // Calculate bulk discount if enabled
        if ($bulkDiscountsEnabled) {
            $bulkDiscounts = get_credit_bulk_discounts();

            if (is_array($bulkDiscounts)) {
                // Sort discounts by quantity in descending order
                usort($bulkDiscounts, function ($a, $b) {
                    return $b['quantity'] - $a['quantity'];
                });

                // Find the highest applicable discount
                foreach ($bulkDiscounts as $tier) {
                    if ($quantity >= $tier['quantity']) {
                        $discountPercentage = $tier['discount'];
                        $discount = $subtotal * ($discountPercentage / 100);
                        break;
                    }
                }
            }
        }

        $total = $subtotal - $discount;

        return [
            'quantity' => $quantity,
            'base_price' => $basePrice,
            'subtotal' => $subtotal,
            'discount_percentage' => $discountPercentage,
            'discount_amount' => $discount,
            'total' => $total,
            'currency' => $currency,
        ];
    }

    /**
     * Validate credit purchase quantity
     */
    public function validatePurchaseQuantity(int $quantity): bool
    {
        $minCredits = get_min_credits_per_purchase();
        $maxCredits = get_max_credits_per_purchase();

        return $quantity >= $minCredits && $quantity <= $maxCredits;
    }

    /**
     * Get available payment methods based on settings
     */
    public function getAvailablePaymentMethods(): array
    {
        $methods = [];

        // Check if Stripe is enabled
        if (is_stripe_enabled()) {
            $methods[] = [
                'id' => PaymentMethod::Stripe->value,
                'name' => 'Credit Card (Stripe)',
                'icon' => 'credit-card',
                'available' => true,
            ];
        }

        // Check if PayPal is enabled
        if (is_paypal_enabled()) {
            $methods[] = [
                'id' => PaymentMethod::Paypal->value,
                'name' => 'PayPal',
                'icon' => 'paypal',
                'available' => true,
            ];
        }

        return $methods;
    }

    /**
     * Create a credit purchase record
     */
    public function createPurchaseRecord(User $user, int $quantity, float $amount, string $currency): Purchase
    {
        $pricing = $this->calculateCreditPrice($quantity);

        return Purchase::create([
            'user_id' => $user->id,
            'item_type' => PurchaseItemType::Credit->value,
            'purchaseable_type' => UserCreditHistory::class,
            'purchaseable_id' => null, // Will be set after UserCreditHistory is created
            'amount' => $pricing['base_price'],
            'currency' => $currency,
            'quantity' => $quantity,
            'total_amount' => $amount,
            'discount_amount' => $pricing['discount_amount'],
            'tax_amount' => 0.00,
            'purchase_date' => now(),
            'notes' => "Credit purchase: {$quantity} credits",
            'created_by' => $user->id,
        ]);
    }

    /**
     * Create a payment record
     */
    public function createPaymentRecord(
        User $user,
        Purchase $purchase,
        string $paymentMethod,
        float $amount,
        string $currency,
        array $metadata = []
    ): Payment {
        return Payment::create([
            'user_id' => $user->id,
            'purchase_id' => $purchase->id,
            'payment_type' => PaymentType::Payment->value,
            'payment_method' => $paymentMethod,
            'amount' => $amount,
            'currency' => $currency,
            'status' => PaymentStatus::Pending->value,
            'payment_date' => now(),
            'service_response' => json_encode($metadata),
            'created_by' => $user->id,
        ]);
    }

    /**
     * Process credit purchase with Stripe
     */
    public function processCreditPurchase(
        User $user,
        int $quantity,
        string $paymentMethod = 'stripe',
        array $paymentData = []
    ): array {
        try {
            // Validate quantity
            if (!$this->validatePurchaseQuantity($quantity)) {
                throw new Exception('Invalid credit quantity');
            }

            // Calculate pricing
            $pricing = $this->calculateCreditPrice($quantity);

            return DB::transaction(function () use ($user, $quantity, $pricing, $paymentMethod, $paymentData) {
                // Create purchase record
                $purchase = $this->purchaseService->createCreditPurchase(
                    $user,
                    $quantity,
                    $pricing['base_price'],
                    $pricing['total'],
                    $pricing['discount_amount'],
                    $pricing['currency']
                );

                // Create payment record
                $payment = $this->paymentService->createPayment(
                    $user,
                    $purchase,
                    $paymentMethod,
                    $pricing['total'],
                    $pricing['currency'],
                    $paymentData
                );

                // Process payment based on method
                if ($paymentMethod === PaymentMethod::Stripe->value) {
                    // Use checkout session for simpler flow (no JavaScript needed)
                    $paymentResult = $this->stripePaymentService->createCheckoutSession($payment);

                    if ($paymentResult['success']) {
                        // Return checkout URL for user to complete payment
                        return [
                            'success' => true,
                            'requires_checkout' => true,
                            'checkout_url' => $paymentResult['checkout_url'],
                            'purchase' => $purchase,
                            'payment' => $payment,
                            'message' => 'Checkout session created successfully',
                        ];
                    } else {
                        // Checkout session creation failed
                        return [
                            'success' => false,
                            'purchase' => $purchase,
                            'payment' => $payment,
                            'message' => $paymentResult['message'] ?? 'Checkout session creation failed',
                        ];
                    }
                } elseif ($paymentMethod === PaymentMethod::Paypal->value) {
                    // Create PayPal order
                    $paymentResult = $this->payPalPaymentService->createOrder($payment);

                    if ($paymentResult['success']) {
                        // Return checkout URL for user to complete payment
                        return [
                            'success' => true,
                            'requires_checkout' => true,
                            'checkout_url' => $paymentResult['checkout_url'],
                            'purchase' => $purchase,
                            'payment' => $payment,
                            'message' => 'PayPal order created successfully',
                        ];
                    } else {
                        // Order creation failed
                        return [
                            'success' => false,
                            'purchase' => $purchase,
                            'payment' => $payment,
                            'message' => $paymentResult['message'] ?? 'PayPal order creation failed',
                        ];
                    }
                } else {
                    throw new Exception('Unsupported payment method: ' . $paymentMethod);
                }
            });
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Credit purchase failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Confirm payment and add credits to user account
     */
    public function confirmPaymentAndAddCredits(string $paymentIntentId): array
    {
        try {
            // Confirm the payment intent with Stripe using Laravel Cashier
            $confirmResult = $this->stripePaymentService->confirmPaymentIntent($paymentIntentId);

            if (!$confirmResult['success']) {
                throw new Exception($confirmResult['message']);
            }

            // Get the payment record
            $payment = $confirmResult['payment'];
            $user = $payment->user;
            $purchase = $payment->purchase;

            // Check if credits were already added
            $existingCreditHistory = UserCreditHistory::where('creditable_type', Purchase::class)
                ->where('creditable_id', $purchase->id)
                ->first();

            if ($existingCreditHistory) {
                return [
                    'success' => true,
                    'purchase' => $purchase,
                    'payment' => $payment,
                    'credits_added' => $purchase->quantity,
                    'credit_history' => $existingCreditHistory,
                    'message' => 'Payment already processed and credits added',
                ];
            }

            // Add credits to user account
            $creditHistory = $this->addCreditsToUser($user, $purchase, $purchase->quantity);

            return [
                'success' => true,
                'purchase' => $purchase,
                'payment' => $payment,
                'credits_added' => $purchase->quantity,
                'credit_history' => $creditHistory,
                'message' => 'Payment confirmed and credits added successfully',
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Payment confirmation failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Handle successful checkout session completion
     */
    public function handleCheckoutSessionCompleted(string $checkoutSessionId): array
    {
        try {
            // Find the payment record by checkout session ID
            $payment = Payment::where('transaction_id', $checkoutSessionId)->first();

            if (!$payment) {
                throw new Exception('Payment record not found for checkout session: ' . $checkoutSessionId);
            }

            // Check if payment is already processed
            if ($payment->status === PaymentStatus::Paid->value) {
                return [
                    'success' => true,
                    'message' => 'Payment already processed',
                    'payment' => $payment,
                ];
            }

            $user = $payment->user;
            $purchase = $payment->purchase;

            // Check if credits were already added
            $existingCreditHistory = UserCreditHistory::where('creditable_type', Purchase::class)
                ->where('creditable_id', $purchase->id)
                ->first();

            if ($existingCreditHistory) {
                // Update payment status to paid
                $payment->update(['status' => PaymentStatus::Paid->value]);

                return [
                    'success' => true,
                    'purchase' => $purchase,
                    'payment' => $payment,
                    'credits_added' => $purchase->quantity,
                    'credit_history' => $existingCreditHistory,
                    'message' => 'Credits already added, payment status updated',
                ];
            }

            // Verify payment status with Stripe to confirm success
            $stripeSessionData = $this->stripePaymentService->retrieveCheckoutSession($checkoutSessionId);

            if (!$stripeSessionData['success']) {
                // Payment verification failed, update status to failed
                $payment->update([
                    'status' => PaymentStatus::Failed->value,
                    'service_response' => json_encode([
                        'checkout_session_id' => $checkoutSessionId,
                        'error' => $stripeSessionData['message'] ?? 'Payment verification failed',
                        'completed_at' => now()->toISOString(),
                    ]),
                ]);

                // Send failure notification
                $user->notify(new \App\Notifications\CreditPurchaseFailed(
                    $purchase,
                    $payment
                ));

                return [
                    'success' => false,
                    'purchase' => $purchase,
                    'payment' => $payment,
                    'message' => 'Payment verification failed: ' . ($stripeSessionData['message'] ?? 'Unknown error'),
                ];
            }

            // Payment verified as successful, add credits to user account
            $creditHistory = $this->addCreditsToUser($user, $purchase, $purchase->quantity);

            // Update payment status to paid
            $payment->update([
                'status' => PaymentStatus::Paid->value,
                'service_response' => json_encode([
                    'checkout_session_id' => $checkoutSessionId,
                    'completed_at' => now()->toISOString(),
                ]),
            ]);

            // Send success notification
            $user->notify(new \App\Notifications\CreditPurchaseSuccessful(
                $purchase,
                $payment,
                $purchase->quantity
            ));

            return [
                'success' => true,
                'purchase' => $purchase,
                'payment' => $payment,
                'credits_added' => $purchase->quantity,
                'credit_history' => $creditHistory,
                'message' => 'Checkout session processed successfully',
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Checkout session processing failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Add credits to user account
     */
    public function addCreditsToUser(User $user, Purchase $purchase, int $quantity): UserCreditHistory
    {
        return DB::transaction(function () use ($user, $purchase, $quantity) {
            // Create credit history record - observer will handle balance calculation and user profile update
            $creditHistory = UserCreditHistory::create([
                'user_id' => $user->id,
                'credit' => $quantity,
                'debit' => 0,
                'balance' => 0, // Will be calculated by observer
                'creditable_type' => Purchase::class,
                'creditable_id' => $purchase->id,
                'created_by' => $user->id,
            ]);

            // Update purchase with purchaseable_id
            $purchase->update(['purchaseable_id' => $creditHistory->id]);

            return $creditHistory;
        });
    }

    /**
     * Deduct credits from user account
     */
    public function deductCreditsFromUser(User $user, int $quantity, string $creditableType, int $creditableId, int $createdBy = null): UserCreditHistory
    {
        return DB::transaction(function () use ($user, $quantity, $creditableType, $creditableId, $createdBy) {
            // Create debit history record - observer will handle balance calculation and user profile update
            $creditHistory = UserCreditHistory::create([
                'user_id' => $user->id,
                'credit' => 0,
                'debit' => $quantity,
                'balance' => 0, // Will be calculated by observer
                'creditable_type' => $creditableType,
                'creditable_id' => $creditableId,
                'created_by' => $createdBy ?? $user->id,
            ]);

            return $creditHistory;
        });
    }

    /**
     * Get current credit balance for user
     */
    public function getUserCreditBalance(User $user): int
    {
        $totalCredits = UserCreditHistory::where('user_id', $user->id)->sum('credit');
        $totalDebits = UserCreditHistory::where('user_id', $user->id)->sum('debit');

        return max(0, $totalCredits - $totalDebits);
    }

    /**
     * Handle successful PayPal payment and add credits
     */
    public function handlePayPalPaymentSuccess(string $orderId): array
    {
        try {
            // Find payment record by order ID
            $payment = Payment::where('transaction_id', $orderId)->first();

            if (!$payment) {
                throw new Exception('Payment record not found for PayPal order: ' . $orderId);
            }

            // Check if payment is already processed
            if ($payment->status === PaymentStatus::Paid->value) {
                // Get existing credit history
                $existingCreditHistory = UserCreditHistory::where('creditable_type', Purchase::class)
                    ->where('creditable_id', $payment->purchase_id)
                    ->first();

                return [
                    'success' => true,
                    'payment' => $payment,
                    'credit_history' => $existingCreditHistory,
                    'message' => 'Payment already processed',
                ];
            }

            $user = $payment->user;
            $purchase = $payment->purchase;

            // Capture payment with PayPal
            $captureResult = $this->payPalPaymentService->capturePayment($orderId);

            if (!$captureResult['success']) {
                // Payment failed, update status
                $payment->update([
                    'status' => PaymentStatus::Failed->value,
                    'service_response' => json_encode([
                        'error' => $captureResult['message'] ?? 'PayPal payment capture failed',
                        'timestamp' => now()->toISOString(),
                    ]),
                ]);

                // Notify user about failed payment
                $user->notify(new \App\Notifications\CreditPurchaseFailed(
                    $purchase,
                    $payment
                ));

                return [
                    'success' => false,
                    'payment' => $payment,
                    'message' => $captureResult['message'] ?? 'PayPal payment capture failed',
                ];
            }

            // Check if credits were already added
            $existingCreditHistory = UserCreditHistory::where('creditable_type', Purchase::class)
                ->where('creditable_id', $purchase->id)
                ->first();

            if ($existingCreditHistory) {
                return [
                    'success' => true,
                    'payment' => $payment,
                    'credit_history' => $existingCreditHistory,
                    'credits_added' => $purchase->quantity,
                    'message' => 'Credits already added for this purchase',
                ];
            }

            // Payment successful, add credits to user account
            $creditHistory = $this->addCreditsToUser($user, $purchase, $purchase->quantity);

            // Send success notification
            $user->notify(new \App\Notifications\CreditPurchaseSuccessful(
                $purchase,
                $payment,
                $purchase->quantity
            ));

            return [
                'success' => true,
                'payment' => $payment,
                'credit_history' => $creditHistory,
                'credits_added' => $purchase->quantity,
                'message' => 'PayPal payment processed and credits added successfully',
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'PayPal payment processing failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get credit purchase history for user
     */
    public function getUserCreditPurchases(User $user, int $limit = 10): array
    {
        $purchases = $user->purchases()
            ->where('item_type', PurchaseItemType::Credit->value)
            ->with(['payments'])
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();

        return $purchases->map(function ($purchase) {
            $firstPayment = $purchase->payments->first();

            return [
                'id' => $purchase->id,
                'quantity' => $purchase->quantity,
                'amount' => $purchase->total_amount,
                'currency' => $purchase->currency,
                'date' => $purchase->purchase_date->format('Y-m-d H:i:s'),
                'status' => $firstPayment ? $firstPayment->status : 'pending',
                'payment_method' => $firstPayment ? $firstPayment->payment_method : 'unknown',
            ];
        })->toArray();
    }
}
